[comment {-*- tcl -*-}]
[vset VERSION 1.2.2]
[manpage_begin struct::record n [vset VERSION]]
[keywords {data structures}]
[keywords record]
[keywords struct]
[copyright {2002, Brett Schwarz <brett_schwarz@yahoo.com>}]
[moddesc   {Tcl Data Structures}]
[titledesc {Define and create records (similar to 'C' structures)}]
[category  {Data structures}]
[require Tcl 8.2]
[require struct::record [opt [vset VERSION]]]
[description]

The [cmd ::struct::record] package provides a mechanism to group
variables together as one data structure, similar to a [term C]
structure. The members of a record can be variables or other
records. However, a record can not contain circular records,
i.e. records that contain the same record as a member.

[para] This package was structured so that it is very similar to how
Tk objects work. Each record definition creates a record object that
encompasses that definition. Subsequently, that record object can
create instances of that record. These instances can then be
manipulated with the [method cget] and [method configure] methods.

[para] The package only contains one top level command, but several
sub commands (see below). It also obeys the namespace in which the
record was defined, hence the objects returned are fully qualified.

[list_begin definitions]

[call [cmd {record define}] [arg recordName] [arg recordMembers] \
     [opt [arg "instanceName1 instanceName2 ..."]]]

Defines a record. [arg recordName] is the name of the record, and is
also used as an object command. This object command is used to create
instances of the record definition. The [arg recordMembers] are the
members of the record that make up the record definition. These are
variables and other records. If optional [arg instanceName] args are
specified, then an instance is generated after the definition is
created for each [arg instanceName].

[call [cmd {record show}] [arg record]]

Returns a list of records that have been defined.

[call [cmd {record show}] [arg instances] [arg recordName]]

Returns the instances that have been instantiated by
[arg recordName].

[call [cmd {record show}] [arg members] [arg recordName]]

Returns the members that are defined for record [arg recordName].
It returns the same format as how the records were defined.

[call [cmd {record show}] [arg values] [arg instanceName]]

Returns a list of values that are set for the instance
[arg instanceName]. The output is a list of key/value pairs. If there
are nested records, then the values of the nested records will
itself be a list.

[call [cmd {record exists}] [arg record] [arg recordName]]

Tests for the existence of a [arg record] with the
name [arg recordName].

[call [cmd {record exists}] [arg instance] [arg instanceName]]

Tests for the existence of a [arg instance] with the
name [arg instanceName].

[call [cmd {record delete}] [arg record] [arg recordName]]

Deletes [arg recordName], and all instances of [arg recordName].
It will return an error if the record does not exist.

[call [cmd {record delete}] [arg instance] [arg instanceName]]

Deletes [arg instance] with the name of [arg instanceName]. It will
return an error if the instance does not exist. Note that this
recursively deletes any nested instances as well.

[list_end]

[section {RECORD MEMBERS}]

Record members can either be variables, or other records, However, the
same record can not be nested witin itself (circular). To define a
nested record, you need to specify the [const record] keyword, along
the with name of the record, and the name of the instance of that
nested record (within the container). For example, it would look like
this:

[para][example_begin]
# this is the nested record
record define mynestedrecord {
    nest1
    nest2
}

# This is the main record
record define myrecord {
    mem1
    mem2
    {record mynestedrecord mem3}
}
[example_end]

You can also assign default or initial values to the members of a
record, by enclosing the member entry in braces:

[para]
[example_begin]
record define myrecord {
    mem1
    {mem2 5}
}
[example_end]

All instances created from this record definition will initially have
[const 5] as the value for member [arg mem2]. If no default is given,
then the value will be the empty string.

[subsection {Getting Values}]

To get a value of a member, there are several ways to do this.

[list_begin definitions]

[call [arg instanceName] [method cget] -[arg member]]

In this form the built-in [method cget] instance method returns the
value of the specified [arg member]. Note the leading dash.

[para] To reach a nested member use [term {dot notation}]:
[example_begin]
[arg instanceName] [method cget] -mem3.nest1
[example_end]


[call [arg instanceName] [method cget] -[arg member1] -[arg member2]]

In this form the built-in [method cget] instance method returns a list
containing the values of both specified members, in the order of specification.

[call [arg instanceName] [method cget]]
[call [arg instanceName] [method configure]]
[call [arg instanceName]]

These forms are all equivalent. They return a dictionary of all
members and the associated values.


[list_end]

[subsection {Setting Values}]

To set a value of a member, there are several ways to do this.

[list_begin definitions]

[call [arg instanceName] [method configure] -[arg member] [arg value]]

In this form the built-in [method configure] instance method sets the
specified [arg member] to the given [arg value]. Note the leading
dash.

[para] To reach a nested member use [term {dot notation}]:
[example_begin]
[arg instanceName] [method configure] -mem3.nest1 value
[example_end]

[call [arg instanceName] [method configure] \
     -[arg member1] [arg value1] \
     -[arg member2] [arg value2]]

In this form the built-in [method configure] instance method sets all
specified members to the associated values.

[list_end]

[subsection {Alias access}]

In the original implementation, access was done by using dot notation
similar to how [term C] structures are accessed. However, there was a
concensus to make the interface more Tcl like, which made sense.

However, the original alias access still exists. It might prove to be
helpful to some.

[para] Basically, for every member of every instance, an alias is
created. This alias is used to get and set values for that member.

An example will illustrate the point, using the above defined records:

[para]
[example_begin]
% # Create an instance first
% myrecord inst1
::inst1

% # To get a member of an instance, just use the alias. It behaves
% # like a Tcl command:
% inst1.mem1

% # To set a member via the alias, just include a value. And optionally
% # the equal sign - syntactic sugar.
% inst1.mem1 = 5
5

% inst1.mem1
5

% # For nested records, just continue with the dot notation.
% # note, no equal sign.
% inst1.mem3.nest1 10
10

% inst1.mem3.nest1
10

% # just the instance by itself gives all member/values pairs for that
% # instance
% inst1
-mem1 5 -mem2 {} -mem3 {-nest1 10 -nest2 {}}

% # and to get all members within the nested record
% inst1.mem3
-nest1 10 -nest2 {}

[example_end]

[section {RECORD COMMAND}]

The following subcommands and corresponding arguments are available to
any record command:

[list_begin definitions]

[call [arg recordName] [arg instanceName]|[const #auto] \
     [opt [arg "-member1 value1 -member2 value2 ..."]]]

Using the [arg recordName] object command that was created from the
record definition, instances of the record definition can be
created.

Once an instance is created, it inherits the members of the record
definition, very similar to how objects work.

During instance generation, an object command for the instance is
created as well, using [arg instanceName].

[para] This object command is used to access the data members of the
instance.

During the instantiation, while values for that instance may be given,
when done, [strong all] values must be given, and be given as
key/value pairs, like for method [method configure]. Nested records
have to be in list format.

[para] Optionally, [const #auto] can be used in place of
[arg instanceName]. When [const #auto] is used, the instance name will
be automatically generated, and of the form

[const recordName[var N]], where [var N] is a unique integer (starting
at 0) that is generated.

[list_end]
[para]

[section {INSTANCE COMMAND}]

The following subcommands and corresponding arguments are available to
any record instance command:

[list_begin definitions]

[call [arg instanceName] [method cget] [opt [arg "-member1 -member2 ..."]]]

Each instance has the method [method cget]. This is very similar to
how Tk widget's [method cget] command works. It queries the values of
the members for that particular instance. If no arguments are given,
then a dictionary is returned.

[call [arg instanceName] [method configure] \
     [opt [arg "-member1 value1 -member2 value2 ..."]]]

Each instance has the method [method configure]. This is very similar
to how Tk widget's [method configure] command works. It sets the
values of the particular members for that particular instance. If no
arguments are given, then a dictionary list is returned.

[list_end]

[section EXAMPLES]

Two examples are provided to give a good illustration on how to use
this package.

[subsection {Example 1 - Contact Information}]

Probably the most obvious example would be to hold contact
information, such as addresses, phone numbers, comments, etc. Since a
person can have multiple phone numbers, multiple email addresses, etc,
we will use nested records to define these. So, the first thing we do
is define the nested records:

[para][example {

##
##  This is an interactive example, to see what is returned by
##  each command as well.
##

% namespace import ::struct::record::*

% # define a nested record. Notice that country has default 'USA'.
% record define locations {
    street
    street2
    city
    state
    zipcode
    {country USA}
    phone
}
::locations
% # Define the main record. Notice that it uses the location record twice.
% record define contacts {
    first
    middle
    last
    {record locations home}
    {record locations work}
}
::contacts
% # Create an instance for the contacts record.
% contacts cont1
::cont1
% # Display some introspection values
% record show records
::contacts ::locations
% #
% record show values cont1
-first {} -middle {} -last {} -home {-street {} -street2 {} -city {} -state {} -zipcode {} -country USA -phone {}} -work {-street {} -street2 {} -city {} -state {} -zipcode {} -country USA -phone {}}
% #
% record show instances contacts
::cont1
% #
% cont1 config
-first {} -middle {} -last {} -home {-street {} -street2 {} -city {} -state {} -zipcode {} -country USA -phone {}} -work {-street {} -street2 {} -city {} -state {} -zipcode {} -country USA -phone {}}
% #
% cont1 cget
-first {} -middle {} -last {} -home {-street {} -street2 {} -city {} -state {} -zipcode {} -country USA -phone {}} -work {-street {} -street2 {} -city {} -state {} -zipcode {} -country USA -phone {}}
% # copy one record to another record
% record define contacts2 [record show members contacts]
::contacts2
% record show members contacts2
first middle last {record locations home} {record locations work}
% record show members contacts
first middle last {record locations home} {record locations work}
%
}]

[subsection {Example 2 - Linked List}]

This next example just illustrates a simple linked list

[para][example {
% # define a very simple record for linked list
% record define linkedlist {
    value
    next
}
::linkedlist
% linkedlist lstart
::lstart
% lstart config -value 1 -next [linkedlist #auto]
% [lstart cget -next] config -value 2 -next [linkedlist #auto]
% [[lstart cget -next] cget -next] config -value 3 -next "end"
% set next lstart
lstart
% while 1 {
    lappend values [$next cget -value]
    set next [$next cget -next]
    if {[string match "end" $next]} break
}
% puts "$values"
1 2 3
% # cleanup linked list
% # We could just use delete record linkedlist also
% foreach I [record show instances linkedlist] {
    record delete instance $I
}
% record show instances linkedlist
%
}]

[vset CATEGORY {struct :: record}]
[include ../common-text/feedback.inc]
[manpage_end]
