package event

import (
	"fmt"
	"time"

	sfxmodel "github.com/signalfx/com_signalfx_metrics_protobuf/model"
)

// Category define how to display the Category.  Category enumerations need to be in sync with sfxmodel
type Category int32

const (
	// USERDEFINED - Created by user via UI or API, e.g. a deployment event
	USERDEFINED Category = 1000000

	// ALERT - Output by anomaly detectors
	ALERT Category = 100000

	// AUDIT - Audit trail events
	AUDIT Category = 200000

	// JOB - Generated by analytics server
	JOB Category = 300000

	// COLLECTD - (deprecated in favor of agent) Event originated within collectd
	COLLECTD Category = 400000

	// SERVICEDISCOVERY - Service discovery event
	SERVICEDISCOVERY Category = 500000

	// EXCEPTION - Created by exception appenders to denote exceptional events
	EXCEPTION Category = 700000

	// AGENT - Event originated within an agent
	AGENT Category = 2000000
)

// An Event is a noteworthy occurrence of something
type Event struct {
	// EventType encodes where the event came from and some of the meaning
	EventType string
	// Category of the event created
	Category Category
	// Dimensions of what is being measured.  They are intrinsic.  Contributes to the identity of
	// the metric. If this changes, we get a new metric identifier
	Dimensions map[string]string
	// Properties is information that's not particularly important to the event, but may be
	// important to the pipeline that uses the event.  They are extrinsic.  It provides additional
	// information about the metric. changes in this set doesn't change the metric identity
	Properties map[string]interface{}
	Timestamp  time.Time
	// Meta is for internal purposes, it does not come in or leave with it.
	Meta map[interface{}]interface{}
}

func (e *Event) String() string {
	return fmt.Sprintf("E[%s\t%v\t%s\t%s\t%s]", e.EventType, e.Category, e.Dimensions, e.Properties, e.Timestamp.String())
}

// New creates a new event with empty meta data
func New(eventType string, category Category, dimensions map[string]string, timestamp time.Time) *Event {
	return NewWithProperties(eventType, category, dimensions, map[string]interface{}{}, timestamp)
}

// ToProtoEC - Converts a protbuf EventCategory to type event.Category
func ToProtoEC(ec sfxmodel.EventCategory) Category {
	response := USERDEFINED
	// Check if the event.Category does not have a corresponding sfxmodel.EventCategory
	if _, ok := sfxmodel.EventCategory_name[int32(ec)]; ok {
		response = Category(ec)
	}
	// Return the sfxmodel.EventCategory
	return response
}

// NewWithProperties creates a new event with passed metadata
func NewWithProperties(eventType string, category Category, dimensions map[string]string, properties map[string]interface{}, timestamp time.Time) *Event {
	return &Event{
		EventType:  eventType,
		Category:   category,
		Dimensions: dimensions,
		Properties: properties,
		Timestamp:  timestamp,
	}
}
