#! /usr/bin/python
# -*- coding: utf-8 -*-

# Copyright (C) 2013 David Callé <davidc@framli.eu>
# This program is free software: you can redistribute it and/or modify it 
# under the terms of the GNU General Public License version 3, as published 
# by the Free Software Foundation.
# 
# This program is distributed in the hope that it will be useful, but 
# WITHOUT ANY WARRANTY; without even the implied warranties of 
# MERCHANTABILITY, SATISFACTORY QUALITY, or FITNESS FOR A PARTICULAR 
# PURPOSE.  See the GNU General Public License for more details.
# 
# You should have received a copy of the GNU General Public License along 
# with this program.  If not, see <http://www.gnu.org/licenses/>.

from gi.repository import Unity, UnityExtras
from gi.repository import Gio, GLib
from launchpadlib.launchpad import Launchpad
import gettext
import datetime

APP_NAME = 'unity-scope-launchpad'
LOCAL_PATH = '/usr/share/locale/'
gettext.bindtextdomain(APP_NAME, LOCAL_PATH)
gettext.textdomain(APP_NAME)
_ = gettext.gettext

GROUP_NAME = 'com.canonical.Unity.Scope.Developer.Launchpad'
UNIQUE_PATH = '/com/canonical/unity/scope/developer/launchpad'
SEARCH_URI = ''
SEARCH_HINT = _('Search Launchpad')
NO_RESULTS_HINT = _('Sorry, there are no Launchpad bugs or projects that match your search.')
PROVIDER_CREDITS = _('Powered by Launchpad')
SVG_DIR = '/usr/share/icons/unity-icon-theme/places/svg/'
PROVIDER_ICON = SVG_DIR+'service-launchpad.svg'
DEFAULT_RESULT_ICON = SVG_DIR+'group-developer.svg'
DEFAULT_RESULT_MIMETYPE = 'text/html'
DEFAULT_RESULT_TYPE = Unity.ResultType.DEFAULT

c1 = {'id'      :'code',
      'name'    :_('Projects'),
      'icon'    :SVG_DIR+'group-developer.svg',
      'renderer':Unity.CategoryRenderer.VERTICAL_TILE}
c2 = {'id'      :'people', 
      'name'    :_('Bugs'),
      'icon'    :SVG_DIR+'group-developer.svg',
      'renderer':Unity.CategoryRenderer.VERTICAL_TILE}
CATEGORIES = [c1, c2]
FILTERS = []
EXTRA_METADATA = []

lp = Launchpad.login_anonymously('unity-lens-launchpad', 'production')

def search(search, filters):
    '''
    Any search method returning results as a list of tuples.
    Available tuple fields:
    uri (string)
    icon (string)
    title (string)
    comment (string)
    dnd_uri (string)
    mimetype (string)
    category (int)
    result_type (Unity ResultType)
    extras metadata fields (variant)
    '''
    results = []
    if len(search) < 2:
        return results
    if search[0] == '#':
        search = search[1:]
    if search.isdigit():
        try:
            bug = lp.bugs[search]
            if bug.title and bug.description:
                results.append({'uri':str(bug.web_link),
                                'category':1,
                                'title':bug.title.encode('utf-8'),
                                'comment':bug.description.encode('utf-8')})
        except Exception as error:
            print (error)
    if not len(search) >= 3 and search.isdigit():
        return results
    for project in lp.projects.search(text=search):
        results.append({'uri':str(project.web_link),
                        'icon':str(project.brand_link),
                        'title':str(project.title.encode('utf-8')),
                        'comment':str(project.name.encode('utf-8'))})
        if len(results) > 25:
            break
    return results


# Classes below this point establish communication
# with Unity, you probably shouldn't modify them.


class MySearch (Unity.ScopeSearchBase):
    def __init__(self, search_context):
        super (MySearch, self).__init__()
        self.set_search_context (search_context)

    def do_run (self):
        '''
        Adds results to the model
        '''
        try:
            result_set = self.search_context.result_set
            for i in search(self.search_context.search_query,
                            self.search_context.filter_state):
                if not 'uri' in i or not i['uri'] or i['uri'] == '':
                    continue
                if not 'icon' in i or not i['icon'] or i['icon'] == '':
                    i['icon'] = DEFAULT_RESULT_ICON
                if not 'mimetype' in i or not i['mimetype'] or i['mimetype'] == '':
                    i['mimetype'] = DEFAULT_RESULT_MIMETYPE
                if not 'result_type' in i or not i['result_type'] or i['result_type'] == '':
                    i['result_type'] = DEFAULT_RESULT_TYPE
                if not 'category' in i or not i['category'] or i['category'] == '':
                    i['category'] = 0
                if not 'title' in i or not i['title']:
                    i['title'] = ''
                if not 'comment' in i or not i['comment']:
                    i['comment'] = ''
                if not 'dnd_uri' in i or not i['dnd_uri'] or i['dnd_uri'] == '':
                    i['dnd_uri'] = i['uri']
                result_set.add_result(**i)
        except Exception as error:
            print (error)

class Preview (Unity.ResultPreviewer):

    def do_run(self):
        project = lp.projects(self.result.comment)
        image = Gio.FileIcon.new(Gio.file_new_for_uri(self.result.icon_hint))
        screenshot = project.screenshots_url
        if screenshot and screenshot != '':
            if screenshot.endswith('.jpg') or screenshot.endswith('.png'):
                image = Gio.FileIcon.new(Gio.file_new_for_uri(screenshot))
        description = project.description
        if description:
            description = description.encode('utf-8')
        else:
            description = ''
        preview = Unity.GenericPreview.new(self.result.title, description, image)
        summary = project.summary
        if summary:
            summary = summary.encode('utf-8')
            preview.props.subtitle = summary
        created = project.date_created
        if created:
            created = created.strftime('%x')
            preview.add_info(Unity.InfoHint.new("created", _("Created"), None, str(created)))

        licenses = project.licenses
        if licenses:
            licenses = ','.join(licenses)
            if ',' in licenses:
                license_display = _("Licenses")
            else:
                license_display = _("License")
            preview.add_info(Unity.InfoHint.new("licenses", license_display, None, licenses))

        programming_language = project.programming_language
        if programming_language:
            programming_language = programming_language.encode('utf-8')
            if ',' in programming_language:
                language_display = _("Languages")
            else:
                language_display = _("Language")
            preview.add_info(Unity.InfoHint.new("languages", language_display, None, programming_language))
        icon = Gio.FileIcon.new (Gio.file_new_for_path(PROVIDER_ICON))
        view_action = Unity.PreviewAction.new("view", _("View"), icon)
        preview.add_action(view_action)
        return preview



class Scope (Unity.AbstractScope):
    def __init__(self):
        Unity.AbstractScope.__init__(self)

    def do_get_search_hint (self):
        return SEARCH_HINT

    def do_get_schema (self):
        '''
        Adds specific metadata fields
        '''
        schema = Unity.Schema.new ()
        if EXTRA_METADATA:
            for m in EXTRA_METADATA:
                schema.add_field(m['id'], m['type'], m['field'])
        #FIXME should be REQUIRED for credits
        schema.add_field('provider_credits', 's', Unity.SchemaFieldType.OPTIONAL)
        return schema

    def do_get_categories (self):
        '''
        Adds categories
        '''
        cs = Unity.CategorySet.new ()
        if CATEGORIES:
            for c in CATEGORIES:
                cat = Unity.Category.new (c['id'], c['name'],
                                          Gio.ThemedIcon.new(c['icon']),
                                          c['renderer'])
                cs.add (cat)
        return cs

    def do_get_filters (self):
        '''
        Adds filters
        '''
        fs = Unity.FilterSet.new ()
#        if FILTERS:
#            
        return fs

    def do_get_group_name (self):
        return GROUP_NAME

    def do_get_unique_name (self):
        return UNIQUE_PATH

    def do_create_search_for_query (self, search_context):
        se = MySearch (search_context)
        return se

    def do_create_previewer(self, result, metadata):
        rp = Preview()
        rp.set_scope_result(result)
        rp.set_search_metadata(metadata)
        return rp

def load_scope():
    return Scope()
