/*
 *  XSMC-CALC: Smith Chart Calculator program for X
 * 
 *
 *      by Lapo Pieri (IK5NAX)  2000-2001
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *   This program is distributed in the hope that it will be useful,
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *   GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 *
 *  Send bugs reports, comments, critique, etc, to ik5nax@amsat.org
 */

#include <stdio.h>
#include <stdlib.h>
#include <unistd.h>
#include <fcntl.h>
#include <ctype.h>
#include <string.h>
#include <errno.h>
#include <math.h>
#include <sys/time.h>
#include <sys/types.h>
#include <sys/stat.h>
#include <X11/Intrinsic.h>
#include <X11/Shell.h>
#include <X11/StringDefs.h>
#include <X11/Xlib.h>
#include <X11/cursorfont.h>
#include <X11/keysym.h>
#include "smc.h"
#include "xsmc.xbm"

Display      *display;
Window       window, root_win;
GC           gc;
Colormap     colormap;
Drawable     drawable;
Pixmap       schart, icon;
Cursor       main_curs;
XEvent       event;
XWindowAttributes  winattr;
char         initflag=0;
XFontStruct  *fontdialogname, *fontchartname;
XWMHints     *xwmhints;

SMCDATA smcdata;
char appname[80];
unsigned long numcolors, colorBLACK, colorWHITE, colorBG,
  colorZ, colorY, colorRHO, colorQ, colorSI, colorDW;
unsigned int wdim=520, scdim;
const float tunesteptab[6]={1.01, 1.02, 1.05, 1.1, 1.2, 1.5};
float tunestep;
int Zcirc=1, Ycirc=0, RHOcirc=0, Qcirc=0;
int main(int argc, char** argv) {
  int clo,  needupdate=1, i, tsidx=3, Keyshift=0, x, y, curswhere=0;
  char b[40], c;
  float re, im, val0, val1, val2;

  scdim=0.7*wdim;
  
  /* Command line scan*/
  if(argc>1) {
    while(1){
      clo=getopt(argc, argv, "h?v");
      if(clo==-1) break;
      switch(clo){
      case 'h':
      case '?':
	printf("\nX Smith Chart Calculator\n");
	printf("for doing calculation on Smith Chart under X\n");
	printf("\n -h\n -? This simple help\n");
	printf(" -v Current version\n\n");
	printf("Released under GNU/GPL by Lapo Pieri (ik5nax@amsat.org)\n\n");
	return(0);
      case 'v':
	printf("\nX Smith Chart Calculator\n");
	printf("version %s\n\n", __VERSION);
	return (0);
      }
    }
  }


  display = XOpenDisplay("");
  if (display == NULL) {
    fprintf(stderr, "Cannot connect to X server.\n");
    return(1);
  }



  /* Init data */
  smcdata.z0=50.; smcdata.f0=1e6;
  smcdata.rerhoSP=0.; smcdata.imrhoSP=0.;
  smcdata.ne=0; smcdata.neidx=0;
  tsidx=3; tunestep=tunesteptab[tsidx];

  root_win=RootWindow(display, DefaultScreen(display));

  createcolors();

  window=XCreateSimpleWindow(display, root_win, 
			     0, 0,
			     wdim, wdim,
			     10,
			     colorBLACK, colorBG
			     );
  XMapWindow(display, window);  
  XStoreName(display, window, "X Smith Chart Calculator");
  main_curs=XCreateFontCursor(display, XC_X_cursor);
  XDefineCursor(display, window, main_curs);

  icon=XCreateBitmapFromData(display, window,
			     icon_bits, icon_width, icon_height);
  if(!icon)
    fprintf(stderr, "Unable to set icon\n");
  else {
    xwmhints=XAllocWMHints();
    if(!xwmhints)
      fprintf(stderr, "Unable to allocate memory for windows hints\n");
    else {
      xwmhints->flags=IconPixmapHint;
      xwmhints->icon_pixmap=icon;
      XSetWMHints(display, window, xwmhints);
      XFree(xwmhints);
    }
  }

  XFlush(display);

  gc=XCreateGC(display, window, 0, 0);

  fontdialogname=XLoadQueryFont(display, "*-helvetica-medium-r-normal--12*");
  if(!fontdialogname) fprintf(stderr, "Unable to load font\n");
  fontchartname=XLoadQueryFont(display, "*helvetica*6*");
  if(!fontchartname) fprintf(stderr, "Unable to load font\n");

  XSelectInput(display, window, 
	       ExposureMask|KeyPressMask|KeyReleaseMask|PointerMotionMask);

  schart=XCreatePixmap(display, DefaultRootWindow(display), 
		       wdim, wdim, 
		       DefaultDepth(display, DefaultScreen(display)));
  
  XSetLineAttributes(display, gc,
		     1, LineSolid, CapButt, JoinBevel);
  
  drawchart();
  XCopyArea(display,schart, window, gc, 0, 0, 
	    wdim, wdim, 0, 0);
  XFlush(display); 

  while(1){
    if(needupdate){
      XCopyArea(display,schart, window, gc, 0, 0, 
		wdim, wdim, 0, 0);
      XFlush(display); 
      needupdate=0;
    }

    XNextEvent(display, &event);
    switch(event.type){
    case Expose:
      XGetWindowAttributes(display, window, &winattr);
      if(winattr.width!=wdim || winattr.height!=wdim) {
       	if(winattr.width<winattr.height) {
	  winattr.height=winattr.width;
	  XResizeWindow(display, window, winattr.width, winattr.height);
	}
	else if(winattr.width>winattr.height) {
	  winattr.width=winattr.height;
	  XResizeWindow(display, window, winattr.width, winattr.height);
	}
	wdim=winattr.width; scdim=0.7*wdim; drawchart();
      }
      needupdate=1;
      break;
      
      
    case KeyPress:

      switch((i=XKeycodeToKeysym(display, event.xkey.keycode,
				 event.xkey.state))) {
	
      case XK_q: 
	/* Exiting */
	XUndefineCursor(display, window);	
	XFreePixmap(display, schart);
	XCloseDisplay(display);
	return 0;
      case XK_1:
	/* Toggle Z circle */
	Zcirc^=1; drawchart();
	needupdate=1; break;
      case XK_2:
	/* Toggle Y circle */
	Ycirc^=1; drawchart();
	needupdate=1; break;
      case XK_3:
	/* Toggle RHO circle */
	RHOcirc^=1; drawchart();
	needupdate=1; break;
      case XK_4:
	/* Toggle Q circle */
	Qcirc^=1; drawchart();
	needupdate=1; break;
      case XK_s:
	/* Insert start point */
	dialogwin("Start Point", "Z Y Rho", &c, b);
	if(entrystringdecode(b, &re, &im)==0) { 
	  switch(c) {
	  case 'z':
	  case 'Z': smcdata.rezSP=re; smcdata.imzSP=im; smcdata.SPtype=c; 
	    break;
	  case 'y':
	  case 'Y': smcdata.reySP=re; smcdata.imySP=im; smcdata.SPtype=c; 
	    break;
	  case 'r':
	  case 'R': smcdata.rerhoSP=re; smcdata.imrhoSP=im; smcdata.SPtype=c; 
	    break;
	  default:
	    fprintf(stderr, "Invalid start point\n");
	  }
	  recalc();
	}
	else 
	  fprintf(stderr, "Invalid start point\n"); 

	drawchart(); needupdate=1;
	break;

      case XK_0:
	/* Insert characteristic impedance (Z0) */
	dialogwin("Z0", "", &c, b);
	if(entrystringdecode(b, &re, &im)==0) {
	  if(re<=0.)  
	    fprintf(stderr, "Invalid characteristic impedance value\n");
	  else {
	    smcdata.z0=re; recalc(); drawchart(); 
	    needupdate=1; }
	}
	else
	  fprintf(stderr, "Invalid characteristic impedance\n");
	break;

      case XK_f:
	/* Insert main frequency (f0) */
	dialogwin("f0", "", &c, b); smcdata.f0=psdec(b);
	recalc(); drawchart(); needupdate=1;
	break;


      case XK_a:
	/* Add a network component */
	if(smcdata.ne<NEL) {
	  dialogwin("Add", "R L C T", &c, b);
	  switch(tolower(c)) {
	  case 'r':
	  case 'l':
	  case 'c':
	  case 't':
	    smcdata.ne++; smcdata.ELtype[smcdata.ne]=c;
	    if(componentvaluestringdecode(b, &val0, &val1, &val2, &c)==0) {
	      smcdata.ELplace[smcdata.ne]=c; smcdata.ELval0[smcdata.ne]=val0;
	      smcdata.ELval1[smcdata.ne]=val1; smcdata.ELval2[smcdata.ne]=val2;
	      recalc();
	    }
	    else
	      smcdata.ne--;
	    break;

	  default:
	    fprintf(stderr, "Unknown component type\n");
	  }
	}
	else
	  fprintf(stderr, "No more space left for component\n");

	smcdata.neidx=smcdata.ne;
	drawchart(); needupdate=1;
	break;

      case XK_i:
	/* Insert a component in the location specified by index */
	if(smcdata.ne<NEL) {
	  /* make place for new network element */
	  for(i=smcdata.ne; i>=smcdata.neidx; i--) {
	  smcdata.ELtype[i+1]=smcdata.ELtype[i];
	  smcdata.ELval0[i+1]=smcdata.ELval0[i];
	  smcdata.ELval1[i+1]=smcdata.ELval1[i];
	  smcdata.ELval2[i+1]=smcdata.ELval2[i];
	  smcdata.ELplace[i+1]=smcdata.ELplace[i];
	  smcdata.rerhoIP[i+1]=smcdata.rerhoIP[i];
	  smcdata.imrhoIP[i+1]=smcdata.imrhoIP[i];
	  }
	  smcdata.ne++;

	  /* and then add the network element in the place just freed */
	  dialogwin("Insert", "R L C T", &c, b);
	  if(tolower(c)=='r' || tolower(c)=='l' || 
	     tolower(c)=='c' || tolower(c)=='t') {
	    smcdata.ELtype[smcdata.neidx]=c;
	    if(componentvaluestringdecode(b, &val0, &val1, &val2, &c)==0) {
	      smcdata.ELplace[smcdata.neidx]=c;
	      smcdata.ELval0[smcdata.neidx]=val0;
	      smcdata.ELval1[smcdata.neidx]=val1;
	      smcdata.ELval2[smcdata.neidx]=val2;
	      recalc();
	      }
	    else 
	      removene();
	  }
	  else {
	  fprintf(stderr, "Unknown component type\n"); removene(); 
	  }

	}
	else
	  fprintf(stderr, "No more space left for component\n");

	drawchart(); needupdate=1; break;

      case XK_d:
	/* Remove a network component */
	removene(); recalc();
	drawchart(); needupdate=1; break;

      case XK_Up:
	if(smcdata.neidx>1) {
	  smcdata.neidx--; drawdata(); needupdate=1; }
	  break;
      case XK_Down: 
	if(smcdata.neidx<smcdata.ne) { 
	  smcdata.neidx++; drawdata(); needupdate=1; }
	break;


      case XK_Shift_L:
      case XK_Shift_R:
	Keyshift=1; break;

      case XK_Right:
	if(Keyshift==0)
	  smcdata.ELval0[smcdata.neidx]*=tunestep; 
	else
	  smcdata.ELval1[smcdata.neidx]*=tunestep; 
	recalc(); drawchart(); needupdate=1; break;
      case XK_Left:
	if(Keyshift==0)
	  smcdata.ELval0[smcdata.neidx]/=tunestep;
	else
	  smcdata.ELval1[smcdata.neidx]/=tunestep;
	recalc(); drawchart(); needupdate=1; break;

      case XK_Page_Up:
	if(tsidx<5) tsidx++; tunestep=tunesteptab[tsidx]; 
	drawdata(); needupdate=1; break;
      case XK_Page_Down:
	if(tsidx>0) tsidx--; tunestep=tunesteptab[tsidx];
	drawdata(); needupdate=1; break;


      case XK_w:
	dialogwin("Write network", "Spice Rfcs", &c, b);
	switch(tolower(c)) {
	case 's':
	  writespice(b); break;
	case 'r':
	  writerfcs(b); break;
	default: fprintf(stderr, "Unknown format\n");
	}
	break;

      case XK_equal:
	if(smcdata.neidx>0) {
	  dialogwin("New val & placement", "", &c, b);
	  if(componentvaluestringdecode(b, &val0, &val1, &val2, &c)==0) {
	    smcdata.ELplace[smcdata.neidx]=c;
	    smcdata.ELval0[smcdata.neidx]=val0;
	    recalc(); drawchart(); needupdate=1;
	  }
	  else { fprintf(stderr, "Invalid value\n"); }
	}	
	break;

      case XK_o:
	optimize(); break;
        
      case XK_p:
	printf("Printing not yet implemented\n"); break;
	
      default:
      }
      break;

    case KeyRelease:

      switch((i=XKeycodeToKeysym(display, event.xkey.keycode,
				 event.xkey.state))) {

      case XK_Shift_L:
      case XK_Shift_R:
	Keyshift=0; break;

      default:
      }      
      break;

    case MotionNotify:
      x=event.xmotion.x; y=event.xmotion.y;
      if(x<scdim+0.02*wdim && y<scdim+0.02*wdim) {
	if(curswhere==0) {
	  main_curs=XCreateFontCursor(display, XC_dotbox);
	  XDefineCursor(display, window, main_curs);
	  curswhere=1;
	}
	x-=0.01*wdim-1; y-=0.01*wdim-1;
	smcdata.rerhoCU=((float)x/(float)scdim-0.5)*2.;
	smcdata.imrhoCU=(0.5-(float)y/(float)scdim)*2.;
	drawdata(); needupdate=1;
      }
      else { 
	if(curswhere==1) {
	  main_curs=XCreateFontCursor(display, XC_X_cursor);
	  XDefineCursor(display, window, main_curs);
	  curswhere=0;
	}
      }
      break;
         
    default:
      
    }
  }

}


void removene(void) {
  int i;
  for(i=smcdata.neidx; i<smcdata.ne; i++) {
    smcdata.ELtype[i]=smcdata.ELtype[i+1];
    smcdata.ELval0[i]=smcdata.ELval0[i+1];
    smcdata.ELval1[i]=smcdata.ELval1[i+1];
    smcdata.ELval2[i]=smcdata.ELval2[i+1];
    smcdata.ELplace[i]=smcdata.ELplace[i+1];
    smcdata.rerhoIP[i]=smcdata.rerhoIP[i+1];
    smcdata.imrhoIP[i]=smcdata.imrhoIP[i+1];
  }
  smcdata.ne--; if(smcdata.neidx>smcdata.ne) smcdata.neidx--;
}


void writespice(char *b) {
  FILE *spf;
  int i, nr=0, nc=0, nl=0, nt=0, nod=0, nos=0, node=0;
  float vr, vi;
  char buff[40];

  if((spf=fopen(b, "w"))==NULL) {
    fprintf(stderr, "Unable to open file %s for writing\n", b);
    return;
  }

  fprintf(spf, "Network generated by X Smith Chart Calculator\n");
  fprintf(spf, "* Check/modify this file before simulation\n\n");

  /* Start impedance is saved as a series of a resistance and a reactance
     and as a parallel network, the last is commented */
  rho2z(smcdata.rerhoIP[0], smcdata.imrhoIP[0], &vr, &vi);
  node++;
  float2prefs(vr*smcdata.z0, buff, 1);
  fprintf(spf, "rsp %d %d %s\n", node-1, node, buff);
  node++;
  if(vi<0.) {
    float2prefs(-1./(2.*M_PI*smcdata.f0*vi*smcdata.z0), buff, 1);
    fprintf(spf, "csp %d %d %s\n", node-1, node, buff);
  }
  else {
    float2prefs(vi*smcdata.z0/(2.*M_PI*smcdata.f0) ,buff, 1);
    fprintf(spf, "lsp %d %d %s\n", node-1, node, buff);
  }

  rho2y(smcdata.rerhoIP[0], smcdata.imrhoIP[0], &vr, &vi);
  float2prefs(smcdata.z0/vr ,buff, 1); 
  fprintf(spf, "* rsp %d %d %s\n", node, 0, buff);
  if(vi>=0.) {
    float2prefs(vi/smcdata.z0/(2.*M_PI*smcdata.f0), buff, 1);
    fprintf(spf, "* csp %d %d %s\n", node, 0, buff);
  }
  else {
    float2prefs(-1./(2.*M_PI*smcdata.f0*vi/smcdata.z0), buff, 1);
    fprintf(spf, "* lsp %d %d %s\n", node, 0, buff);
  }

  for(i=1; i<=smcdata.ne; i++) {
    switch(tolower(smcdata.ELtype[i])) {
    case 'r': 
      nr++; 
      if(smcdata.ELplace[i]=='s') { 
	node++; float2prefs(smcdata.ELval0[i], buff, 1);
	fprintf(spf, "r%d %d %d %s\n", nr, node-1, node, buff); 
      }
      else {
	float2prefs(smcdata.ELval0[i], buff, 1);
	fprintf(spf, "r%d %d %d %s\n", nr, node, 0, buff); 
      }
      break;
    case 'l': 
      nl++;
      if(smcdata.ELplace[i]=='s') {
	node++; float2prefs(smcdata.ELval0[i], buff, 1);
	fprintf(spf, "l%d %d %d %s\n", nl, node-1, node, buff);
      }
      else {
	float2prefs(smcdata.ELval0[i], buff, 1);
	fprintf(spf, "l%d %d %d %s\n", nl, node, 0, buff);
      }
      break;
    case 'c': 
      nc++;
      if(smcdata.ELplace[i]=='s') {
	node++; float2prefs(smcdata.ELval0[i], buff, 1);
	fprintf(spf, "c%d %d %d %s\n", nc, node-1, node, buff);
      }
      else {
	float2prefs(smcdata.ELval0[i], buff, 1);
	fprintf(spf, "c%d %d %d %s\n", nc, node, 0, buff);
      }
      break;
    case 't':
      nt++;
      switch(tolower(smcdata.ELplace[i])) {
      case 't':
	node++; float2prefs(smcdata.ELval1[i], buff, 1);
	fprintf(spf, "t%d %d %d %d %d z0=%s ", nt, node-1, 0, node, 0, buff);
	float2prefs(smcdata.f0, buff, 1);
	fprintf(spf, "f=%s nl=%f\n", buff,  smcdata.ELval0[i]/360.);
	break;
      case 'o':	
	nod+=2; float2prefs(smcdata.ELval1[i], buff, 1);
	fprintf(spf, "t%d %d %d %dd %dd z0=%s ", 
		nt, node, 0, nod-1, nod, buff);
	float2prefs(smcdata.f0, buff, 1);
	fprintf(spf, "f=%s nl=%f\n", buff,  smcdata.ELval0[i]/360.);
	break;
      case 'c': 
	nos++; float2prefs(smcdata.ELval1[i], buff, 1);
	fprintf(spf, "t%d %d %d %dd %dd z0=%s ", nt, node, 0, nos, nos, buff);
	float2prefs(smcdata.f0, buff, 1);
	fprintf(spf, "f=%s nl=%f\n", buff,  smcdata.ELval0[i]/360.);
	break;
      case 'l':	
	node++; nos++; float2prefs(smcdata.ELval1[i], buff, 1);
	fprintf(spf, "t%d %d %d %dd %dd z0=%s ",
		nt, node-1, node, nos, nos, buff);
	float2prefs(smcdata.f0, buff, 1);
	fprintf(spf, "f=%s nl=%f\n", buff,  smcdata.ELval0[i]/360.);
	break;
      case 'a': 
	node++; nos+=2; float2prefs(smcdata.ELval1[i], buff, 1);
	fprintf(spf, "t%d %d %d %dd %dd z0=%s ",
		nt, node-1, node, nos-1, nos, buff);
	float2prefs(smcdata.f0, buff, 1);
	fprintf(spf, "f=%s nl=%f\n", buff,  smcdata.ELval0[i]/360.);
	break;

      }
      break;
    default: fprintf(spf, "* Unknown component '%c'\n", smcdata.ELtype[i]);
    }
  }

  node++;
  float2prefs(smcdata.z0, buff, 1);
  fprintf(spf, "r0 %d %d %s\n", node-1, node, buff);
  fprintf(spf, "v1 %d %d dc 0 ac 1\n", node, 0);

  float2prefs(smcdata.f0, buff, 1);
  fprintf(spf, "\n.ac lin 1 %s %s\n", buff, buff);
  fprintf(spf, ".end\n");
  fclose(spf);
}


void writerfcs(char *b) {
  fprintf(stderr, "Unsupported yet\n");
}


void optimize(void) {
  int i=1;
  float diff, olddiff;


  olddiff=(smcdata.rerhoIP[smcdata.ne]-50.)*(smcdata.rerhoIP[smcdata.ne]-50.)+
    (smcdata.imrhoIP[smcdata.ne]-0.)*(smcdata.imrhoIP[smcdata.ne]-0.);

  while(1) {

    /* Check if user stop optimization with ESC */
    if(XCheckWindowEvent(display, window, KeyPressMask, &event)) {
      if(XKeycodeToKeysym(display, event.xkey.keycode,
			  event.xkey.state)==XK_Escape)
	return;
    }

    /* Search for better value, that's the optimization core */
    smcdata.ELval0[i]*=1.01;
    recalc(); 
    diff=(smcdata.rerhoIP[smcdata.ne]-50.)*(smcdata.rerhoIP[smcdata.ne]-50.)+
      (smcdata.imrhoIP[smcdata.ne]-0.)*(smcdata.imrhoIP[smcdata.ne]-0.);
    
    if(olddiff<diff) {
      smcdata.ELval0[i]/=1.01;
    }

    if(fabs(olddiff-diff)<1e-6) break;

    /* Redraw new data */
    drawchart();
    XCopyArea(display,schart, window, gc, 0, 0, 
	      wdim, wdim, 0, 0);
    XFlush(display); 

    /* Increse element index involved in optimization */
    if(i<smcdata.ne) i++;
    else i=1;
  }

}

void createcolors(void){
XColor       color;

  colormap=DefaultColormap(display, 0);
  color.flags=DoRed|DoGreen|DoBlue;
  
  /* black */
  color.red=0; color.green=0; color.blue=0;
  XAllocColor(display, colormap, &color);
  colorBLACK=color.pixel;
  
  /* white */
  color.red=65535; color.green=65535; color.blue=65535;
  XAllocColor(display, colormap, &color);
  colorWHITE=color.pixel;

  /* Background (wheat)*/
  color.red=63000; color.green=58000; color.blue=48000;
  XAllocColor(display, colormap, &color);
  colorBG=color.pixel;
  
  /* Impedance (red) */
  color.red=65535; color.green=0; color.blue=0;
  XAllocColor(display, colormap, &color);
  colorZ=color.pixel;

  /* Admittance (blue) */
  color.red=0; color.green=0; color.blue=65535;
  XAllocColor(display, colormap, &color);
  colorY=color.pixel;

  /* Reflection Coefficient Rho (green) */
  color.red=0; color.green=50000; color.blue=0;
  XAllocColor(display, colormap, &color);
  colorRHO=color.pixel;

  /* Q (yellow) */
  color.red=65535; color.green=65535; color.blue=0;
  XAllocColor(display, colormap, &color);
  colorQ=color.pixel;

  /* Start Impedance (purple) */
  color.red=50000; color.green=0; color.blue=50000;
  XAllocColor(display, colormap, &color);
  colorSI=color.pixel;

  /* Dialog Window () */
  color.red=33000; color.green=65000; color.blue=54000;
  XAllocColor(display, colormap, &color);
  colorDW=color.pixel;
}

