/**
 * @file    scale_panel.c
 * @brief   Routines for rescaling a plot via an interface window.
 *
 * @author  Denis Pollney
 * @date    23 Aug 2003
 *
 * @par Copyright (C) 2001-2002 Denis Pollney
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2, or (at your option)
 *  any later version.
 * @par
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details
 * @par
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 */

#include "ygraph.h"

/**
 * @brief    Set the range based on the value contained in a dialog entry.
 *
 * @param    entry     The entry field containing the scale value.
 * @param    data_set  The data set to be re-scaled.
 *
 * @returns  True, if everything went well.
 */
gboolean
scale_set_from_entry(GtkEntry* entry, DataSet* data_set)
{
  Plot* plot;
  gdouble scale;
  guint i;

  scale = g_strtod(gtk_entry_get_text(entry), NULL);

  dataset_calc_rescale(data_set, scale);

  for (i=0; i<global_plot_window->len; ++i)
    {
      plot = g_array_index(global_plot_window, Plot*, i);
      plot_range_set(plot, INIT_DATA);
      plot->legend->rebuild = TRUE;
      plot_window_reconfigure(plot);
      plot_window_display_all(plot);
    }

  return TRUE;
}

/**
 * @brief    Apply the scale modifications.
 *
 * @param    panel             The scale entry panel.
 */
gboolean
scale_panel_apply(GtkWidget* panel)
{
  DataSet* data_set;
  gchar* scale_str;
  gdouble scale;
  guint i;

  UNUSED(panel);
  for (i=0; i<global_data_set_list->len; ++i)
    {
      data_set = g_array_index(global_data_set_list, DataSet*, i);
      scale_str = gtk_entry_get_text(GTK_ENTRY(data_set->scale_entry_field));
      scale = g_strtod(scale_str, NULL);
      
      if (scale != data_set->scale)
        scale_set_from_entry(data_set->scale_entry_field, data_set);
    }

  return TRUE;
}

/**
 * @brief    Destroy the scale panel and reset the global pointer.
 *
 * @param    panel             The scale entry panel.
 */
gboolean
scale_panel_destroy(GtkWidget* panel)
{
  UNUSED(panel);
  gtk_widget_destroy(global_scale_panel);
  global_scale_panel = NULL;
  return TRUE;
}

/**
 * @brief    Apply the scale modifications and get rid of scale panel.
 *
 * @param    panel             The scale entry panel.
 */
gboolean
scale_panel_okay(GtkWidget* panel)
{
  scale_panel_apply(panel);
  scale_panel_destroy(panel);
  return TRUE;
}

/**
 * @brief    Draw the scale panel window.
 */
void
scale_panel_create(Plot* plot, gint action, GtkItem* range_set_button)
{
  GtkWidget* dialog;
  GtkWidget* scale_table;
  GtkWidget* scale_title_label;
  GtkWidget* scale_label;
  GtkWidget* scale_entry;
  GtkWidget* okay_button;
  GtkWidget* apply_button;
  GtkWidget* cancel_button;
  DataSet* data_set;
  gchar scale_str[SCALE_ENTRY_SIZE];
  guint i;

  UNUSED(plot);
  UNUSED(action);
  UNUSED(range_set_button);
  if (global_data_set_list->len == 0)
    return;

  if (global_scale_panel != NULL)
    {
      gtk_widget_destroy(global_scale_panel);
      global_scale_panel = NULL;
    }

  /*
   * Set up scale panel window.
   */
  dialog = gtk_dialog_new();
  gtk_window_set_title(GTK_WINDOW(dialog), SCALE_PANEL_NAME);
  gtk_window_set_policy(GTK_WINDOW(dialog), FALSE, FALSE, TRUE);

  scale_table = gtk_table_new(global_data_set_list->len, 2, TRUE);

  scale_title_label = gtk_label_new(SCALE_TITLE_STRING);
  gtk_widget_show(scale_title_label);
  gtk_table_attach_defaults(GTK_TABLE(scale_table), scale_title_label,
                            1, 2, 0, 1);

  /*
   * Set up a scale entry field for each data set.
   */
  for (i=0; i<global_data_set_list->len; ++i)
    {
      data_set = g_array_index(global_data_set_list, DataSet*, i);
      scale_label = gtk_label_new(data_set->name);
      gtk_widget_show(scale_label);
      gtk_table_attach_defaults(GTK_TABLE(scale_table), scale_label,
                                0, 1, i+1, i+2);

      if (data_set->scale_entry_field != NULL)
        {
          gtk_widget_destroy(GTK_WIDGET(data_set->scale_entry_field));
          data_set->scale_entry_field = NULL;
        }

      scale_entry = gtk_entry_new();
      gtk_entry_set_max_length(GTK_ENTRY(scale_entry), SCALE_ENTRY_SIZE);
      g_snprintf(scale_str, SCALE_ENTRY_SIZE, "%g", data_set->scale);
      gtk_entry_set_text(GTK_ENTRY(scale_entry), scale_str);
      gtk_signal_connect(GTK_OBJECT(scale_entry), "activate",
                                    GTK_SIGNAL_FUNC(scale_set_from_entry),
                                    data_set);
      gtk_widget_show(scale_entry);
      gtk_table_attach_defaults(GTK_TABLE(scale_table), scale_entry,
                                1, 2, i+1, i+2);
      data_set->scale_entry_field = GTK_ENTRY(scale_entry);
    }
  
  gtk_widget_show(scale_table);
  gtk_box_pack_start(GTK_BOX(GTK_DIALOG(dialog)->vbox), scale_table, FALSE,
                     FALSE, 0);
  /*
   * Okay button.
   */
  okay_button = gtk_button_new_with_label(OKAY_BUTTON_LABEL);
  gtk_signal_connect (GTK_OBJECT (okay_button), "clicked",
                      GTK_SIGNAL_FUNC (scale_panel_okay), NULL);
  gtk_container_add (GTK_CONTAINER (GTK_DIALOG(dialog)->action_area),
                     okay_button);
  gtk_widget_show(okay_button);

  /*
   * Apply button.
   */
  apply_button = gtk_button_new_with_label(APPLY_BUTTON_LABEL);
  gtk_signal_connect(GTK_OBJECT(apply_button), "clicked",
                     GTK_SIGNAL_FUNC(scale_panel_apply), NULL);
  gtk_container_add (GTK_CONTAINER (GTK_DIALOG(dialog)->action_area),
                     apply_button);
  gtk_widget_show(apply_button);

  /*
   * Cancel button.
   */
  cancel_button = gtk_button_new_with_label(CANCEL_BUTTON_LABEL);
  gtk_signal_connect (GTK_OBJECT (cancel_button), "clicked",
                      GTK_SIGNAL_FUNC (scale_panel_destroy),
                      global_scale_panel);
  gtk_container_add (GTK_CONTAINER (GTK_DIALOG(dialog)->action_area),
                     cancel_button);
  gtk_widget_show(cancel_button);


  global_scale_panel = dialog;
  gtk_widget_show_all (global_scale_panel);
}
