﻿// Licensed to the .NET Foundation under one or more agreements.
// The .NET Foundation licenses this file to you under the MIT license.

using System.Collections;
using System.Collections.Concurrent;
using System.Collections.Generic;
using System.Collections.Immutable;
using System.Collections.ObjectModel;
using System.Collections.Specialized;
using System.Text.Json.Serialization;
using System.Text.Json.Serialization.Tests;
using System.Threading.Tasks;
using Xunit;

namespace System.Text.Json.SourceGeneration.Tests
{
    public partial class CollectionTests_Metadata : CollectionTests
    {
        public CollectionTests_Metadata()
            : this(new StringSerializerWrapper(CollectionTestsContext_Metadata.Default, (options) => new CollectionTestsContext_Metadata(options)))
        {
        }

        protected CollectionTests_Metadata(JsonSerializerWrapperForString serializerWrapper)
            : base(serializerWrapper, null)
        {
        }

        [Fact]
        public override async Task ReadSimpleKeyValuePairPartialData()
        {
            KeyValuePair<string, int> kvp = await JsonSerializerWrapperForString.DeserializeWrapper<KeyValuePair<string, int>>(@"{""Key"": ""123""}");
            Assert.Equal("123", kvp.Key);
            Assert.Equal(0, kvp.Value);

            kvp = await JsonSerializerWrapperForString.DeserializeWrapper<KeyValuePair<string, int>>(@"{""Key"": ""Key"", ""Value"": 123, ""Value2"": 456}");
            Assert.Equal("Key", kvp.Key);
            Assert.Equal(123, kvp.Value);

            kvp = await JsonSerializerWrapperForString.DeserializeWrapper<KeyValuePair<string, int>>(@"{""Key"": ""Key"", ""Val"": 123}");
            Assert.Equal("Key", kvp.Key);
            Assert.Equal(0, kvp.Value);
        }

        [Fact]
        public override async Task HonorCLRProperties()
        {
            var options = new JsonSerializerOptions
            {
                PropertyNamingPolicy = new LeadingUnderscorePolicy() // Key -> _Key, Value -> _Value
            };

            // Since object converter (not KVP converter) is used, payloads not compliant with naming policy won't yield matches.
            string json = @"{""Key"":""Hello, World!"",""Value"":1}";
            KeyValuePair<string, int> kvp = await JsonSerializerWrapperForString.DeserializeWrapper<KeyValuePair<string, int>>(json, options);
            Assert.Null(kvp.Key);
            Assert.Equal(0, kvp.Value);

            // "Key" and "Value" matching is case sensitive.
            json = @"{""key"":""Hello, World!"",""value"":1}";
            kvp = await JsonSerializerWrapperForString.DeserializeWrapper<KeyValuePair<string, int>>(json, options);
            Assert.Null(kvp.Key);
            Assert.Equal(0, kvp.Value);

            // "Key" and "Value" matching is case sensitive, even when case insensitivity is on.
            // Case sensitivity only applies to the result of converting the CLR property names
            // (Key -> _Key, Value -> _Value) with the naming policy.
            options = new JsonSerializerOptions
            {
                PropertyNamingPolicy = new LeadingUnderscorePolicy(),
                PropertyNameCaseInsensitive = true
            };

            kvp = await JsonSerializerWrapperForString.DeserializeWrapper<KeyValuePair<string, int>>(json, options);
            Assert.Null(kvp.Key);
            Assert.Equal(0, kvp.Value);
        }

        [Fact]
        public override async Task HonorNamingPolicy_CaseInsensitive()
        {
            const string json = @"{""key"":""Hello, World!"",""value"":1}";

            // Baseline - with case-sensitive matching, the payload doesn't have mapping properties.
            KeyValuePair<string, int> kvp = await JsonSerializerWrapperForString.DeserializeWrapper<KeyValuePair<string, int>>(json);
            Assert.Null(kvp.Key);
            Assert.Equal(0, kvp.Value);

            // Test - with case-insensitivity on, we have property matches.
            var options = new JsonSerializerOptions
            {
                PropertyNameCaseInsensitive = true
            };

            kvp = await JsonSerializerWrapperForString.DeserializeWrapper<KeyValuePair<string, int>>(json, options);
            Assert.Equal("Hello, World!", kvp.Key);
            Assert.Equal(1, kvp.Value);
        }

        [JsonSourceGenerationOptions(GenerationMode = JsonSourceGenerationMode.Metadata)]
        [JsonSerializable(typeof(ConcurrentDictionary<string, string>))]
        [JsonSerializable(typeof(ConcurrentQueue<string>))]
        [JsonSerializable(typeof(ConcurrentStack<string>))]
        [JsonSerializable(typeof(BlockingCollection<string>))]
        [JsonSerializable(typeof(ConcurrentBag<string>))]
        [JsonSerializable(typeof(GenericConcurrentQueuePrivateConstructor<string>))]
        [JsonSerializable(typeof(GenericConcurrentQueueInternalConstructor<string>))]
        [JsonSerializable(typeof(GenericConcurrentStackPrivateConstructor<string>))]
        [JsonSerializable(typeof(GenericConcurrentStackInternalConstructor<string>))]
        [JsonSerializable(typeof(IDictionary))]
        [JsonSerializable(typeof(Dictionary<string, string>))]
        [JsonSerializable(typeof(SortedDictionary<string, string>))]
        [JsonSerializable(typeof(object))]
        [JsonSerializable(typeof(IDictionary<string, string>))]
        [JsonSerializable(typeof(IReadOnlyDictionary<string, string>))]
        [JsonSerializable(typeof(ImmutableDictionary<string, string>))]
        [JsonSerializable(typeof(IImmutableDictionary<string, string>))]
        [JsonSerializable(typeof(ImmutableSortedDictionary<string, string>))]
        [JsonSerializable(typeof(Hashtable))]
        [JsonSerializable(typeof(SortedList))]
        [JsonSerializable(typeof(WrapperForIDictionary))]
        [JsonSerializable(typeof(StringToStringDictionaryWrapper))]
        [JsonSerializable(typeof(StringToStringSortedDictionaryWrapper))]
        [JsonSerializable(typeof(GenericIDictionaryWrapper<string, string>))]
        [JsonSerializable(typeof(GenericIReadOnlyDictionaryWrapper<string, string>))]
        [JsonSerializable(typeof(StringToStringIImmutableDictionaryWrapper))]
        [JsonSerializable(typeof(HashtableWrapper))]
        [JsonSerializable(typeof(SortedListWrapper))]
        [JsonSerializable(typeof(GenericStructIDictionaryWrapper<string, string>))]
        [JsonSerializable(typeof(GenericStructIDictionaryWrapper<string, string>?))]
        [JsonSerializable(typeof(StructWrapperForIDictionary))]
        [JsonSerializable(typeof(Dictionary<string, object>))]
        [JsonSerializable(typeof(IDictionary<string, object>))]
        [JsonSerializable(typeof(GenericIDictionaryWrapper<string, object>))]
        [JsonSerializable(typeof(GenericIDictionaryWrapper<string, string>))]
        [JsonSerializable(typeof(PocoDictionary))]
        [JsonSerializable(typeof(IDictionary<string, List<int>>))]
        [JsonSerializable(typeof(ImmutableDictionary<string, List<int>>))]
        [JsonSerializable(typeof(IImmutableDictionary<string, List<int>>))]
        [JsonSerializable(typeof(Dictionary<string, int[]>))]
        [JsonSerializable(typeof(List<Dictionary<string, int>>))]
        [JsonSerializable(typeof(List<ImmutableSortedDictionary<string, int>>))]
        [JsonSerializable(typeof(Dictionary<string, int>[]), TypeInfoPropertyName = "ArrayOfDictionaryStringInt")]
        [JsonSerializable(typeof(ImmutableSortedDictionary<string, int>[]))]
        [JsonSerializable(typeof(Dictionary<string, Dictionary<string, int>>))]
        [JsonSerializable(typeof(ImmutableSortedDictionary<string, ImmutableSortedDictionary<string, int>>))]
        [JsonSerializable(typeof(Dictionary<string, Dictionary<string, Dictionary<string, int>>>))]
        [JsonSerializable(typeof(Dictionary<string, Dictionary<string, int>[]>))]
        [JsonSerializable(typeof(SimpleTestClass))]
        [JsonSerializable(typeof(Dictionary<string, int>))]
        [JsonSerializable(typeof(Dictionary<string, JsonElement>))]
        [JsonSerializable(typeof(PocoDuplicate))]
        [JsonSerializable(typeof(ClassWithIgnoredDictionary1))]
        [JsonSerializable(typeof(ClassWithIgnoredDictionary2))]
        [JsonSerializable(typeof(ClassWithIgnoredDictionary3))]
        [JsonSerializable(typeof(ClassWithIgnoredDictionary4))]
        [JsonSerializable(typeof(ClassWithIgnoredDictionary5))]
        [JsonSerializable(typeof(ClassWithIgnoredDictionary6))]
        [JsonSerializable(typeof(ClassWithIgnoredDictionary7))]
        [JsonSerializable(typeof(ClassWithIgnoredIDictionary))]
        [JsonSerializable(typeof(ClassWithIgnoreAttributeDictionary))]
        [JsonSerializable(typeof(ClassWithIgnoredImmutableDictionary))]
        [JsonSerializable(typeof(Dictionary<string, AllSingleUpperProperties_Child>))]
        [JsonSerializable(typeof(AllSingleUpperPropertiesParent))]
        [JsonSerializable(typeof(ClassWithDictionaryOfString_ChildWithDictionaryOfString))]
        [JsonSerializable(typeof(ClassWithDictionaryOfString_ChildWithDictionaryOfString))]
        [JsonSerializable(typeof(ClassWithDictionaryAndProperty_DictionaryFirst))]
        [JsonSerializable(typeof(ClassWithDictionaryAndProperty_DictionaryLast))]
        [JsonSerializable(typeof(SimpleClassWithDictionaries))]
        [JsonSerializable(typeof(DictionaryThatOnlyImplementsIDictionaryOfStringTValue<int>))]
        [JsonSerializable(typeof(DictionaryThatOnlyImplementsIDictionaryOfStringTValue<Poco>))]
        [JsonSerializable(typeof(DictionaryThatOnlyImplementsIDictionaryOfStringPoco))]
        [JsonSerializable(typeof(DictionaryThatHasIncompatibleEnumerator))]
        [JsonSerializable(typeof(Dictionary<string, ClassWithoutParameterlessCtor>))]
        [JsonSerializable(typeof(StructWrapperForIDictionary?))]
        [JsonSerializable(typeof(ClassWithStructIDictionaryWrapper))]
        [JsonSerializable(typeof(Poco))]
        [JsonSerializable(typeof(JsonElement))]
        [JsonSerializable(typeof(string))]
        [JsonSerializable(typeof(IDictionary<int, int>))]
        [JsonSerializable(typeof(Dictionary<string, ClassWithInternalParameterlessConstructor>))]
        [JsonSerializable(typeof(Dictionary<string, ClassWithPrivateParameterlessConstructor>))]
        [JsonSerializable(typeof(Dictionary<string, Dictionary<string, CustomClass>>))]
        [JsonSerializable(typeof(TestClassWithDictionary))]
        [JsonSerializable(typeof(IReadOnlyDictionary<Uri, int>))]
        [JsonSerializable(typeof(GenericIReadOnlyDictionaryWrapper<Uri, int>))]
        [JsonSerializable(typeof(List<int>))]
        [JsonSerializable(typeof(IReadOnlyDictionary<int, int>))]
        [JsonSerializable(typeof(Dictionary<string, CustomClass>))]
        [JsonSerializable(typeof(IDerivedIDictionaryOfTKeyTValue<string, string>))]
        [JsonSerializable(typeof(List<List<int>>))]
        [JsonSerializable(typeof(GenericListWrapper<StringListWrapper>))]
        [JsonSerializable(typeof(List<List<int>>))]
        [JsonSerializable(typeof(GenericListWrapper<StringListWrapper>))]
        [JsonSerializable(typeof(List<int[]>))]
        [JsonSerializable(typeof(GenericListWrapper<string[]>))]
        [JsonSerializable(typeof(List<int>[]), TypeInfoPropertyName = "ArrayOfIntList")]
        [JsonSerializable(typeof(StringListWrapper[]))]
        [JsonSerializable(typeof(List<int>))]
        [JsonSerializable(typeof(StringListWrapper))]
        [JsonSerializable(typeof(IEnumerable<IEnumerable<int>>))]
        [JsonSerializable(typeof(GenericIEnumerableWrapper<StringIEnumerableWrapper>))]
        [JsonSerializable(typeof(IEnumerable<int[]>))]
        [JsonSerializable(typeof(GenericIEnumerableWrapper<int[]>))]
        [JsonSerializable(typeof(IEnumerable<int>[]), TypeInfoPropertyName = "ArrayOfIntIEnumerable")]
        [JsonSerializable(typeof(StringIEnumerableWrapper[]))]
        [JsonSerializable(typeof(IEnumerable<int>))]
        [JsonSerializable(typeof(StringIEnumerableWrapper))]
        [JsonSerializable(typeof(IList<IList<int>>))]
        [JsonSerializable(typeof(GenericIListWrapper<StringIListWrapper>))]
        [JsonSerializable(typeof(IList<int[]>))]
        [JsonSerializable(typeof(GenericIListWrapper<string[]>))]
        [JsonSerializable(typeof(IList<int>[]), TypeInfoPropertyName = "ArrayOfIntIList")]
        [JsonSerializable(typeof(StringIListWrapper[]))]
        [JsonSerializable(typeof(IList<int>))]
        [JsonSerializable(typeof(StringIListWrapper))]
        [JsonSerializable(typeof(GenericStructIListWrapper<int>))]
        [JsonSerializable(typeof(GenericStructIListWrapper<int>?))]
        [JsonSerializable(typeof(GenericStructICollectionWrapper<int>))]
        [JsonSerializable(typeof(GenericStructICollectionWrapper<int>?))]
        [JsonSerializable(typeof(ICollection<ICollection<int>>))]
        [JsonSerializable(typeof(ICollection<int[]>))]
        [JsonSerializable(typeof(GenericICollectionWrapper<string[]>))]
        [JsonSerializable(typeof(ICollection<int>[]), TypeInfoPropertyName = "ArrayOfIntICollection")]
        [JsonSerializable(typeof(ICollection<int>))]
        [JsonSerializable(typeof(GenericICollectionWrapper<string>))]
        [JsonSerializable(typeof(IReadOnlyCollection<IReadOnlyCollection<int>>))]
        [JsonSerializable(typeof(GenericIReadOnlyCollectionWrapper<WrapperForIReadOnlyCollectionOfT<string>>))]
        [JsonSerializable(typeof(IReadOnlyCollection<int[]>))]
        [JsonSerializable(typeof(GenericIReadOnlyCollectionWrapper<int[]>))]
        [JsonSerializable(typeof(IReadOnlyCollection<int>[]), TypeInfoPropertyName = "ArrayOfIntIReadOnlyCollection")]
        [JsonSerializable(typeof(WrapperForIReadOnlyCollectionOfT<string>[]))]
        [JsonSerializable(typeof(IReadOnlyCollection<int>))]
        [JsonSerializable(typeof(WrapperForIReadOnlyCollectionOfT<string>))]
        [JsonSerializable(typeof(IReadOnlyList<IReadOnlyList<int>>))]
        [JsonSerializable(typeof(GenericIReadOnlyListWrapper<StringIReadOnlyListWrapper>))]
        [JsonSerializable(typeof(IReadOnlyList<int[]>))]
        [JsonSerializable(typeof(GenericIReadOnlyListWrapper<string[]>))]
        [JsonSerializable(typeof(IReadOnlyList<int>[]), TypeInfoPropertyName = "ArrayOfIntIReadOnlyList")]
        [JsonSerializable(typeof(StringIReadOnlyListWrapper[]))]
        [JsonSerializable(typeof(IReadOnlyList<int>))]
        [JsonSerializable(typeof(StringIReadOnlyListWrapper))]
        [JsonSerializable(typeof(ISet<ISet<int>>))]
        [JsonSerializable(typeof(GenericISetWrapper<StringISetWrapper>))]
        [JsonSerializable(typeof(GenericStructISetWrapper<int>))]
        [JsonSerializable(typeof(GenericStructISetWrapper<int>?))]
        [JsonSerializable(typeof(ISet<HashSet<int>>))]
        [JsonSerializable(typeof(HashSet<ISet<int>>))]
        [JsonSerializable(typeof(ISet<int[]>))]
        [JsonSerializable(typeof(ISet<int>[]), TypeInfoPropertyName = "ArrayOfIntISet")]
        [JsonSerializable(typeof(ISet<int>))]
        [JsonSerializable(typeof(Stack<Stack<int>>))]
        [JsonSerializable(typeof(GenericStackWrapper<StringStackWrapper>))]
        [JsonSerializable(typeof(Stack<int[]>))]
        [JsonSerializable(typeof(GenericStackWrapper<string[]>))]
        [JsonSerializable(typeof(Stack<int>[]), TypeInfoPropertyName = "ArrayOfIntStack")]
        [JsonSerializable(typeof(StringStackWrapper[]))]
        [JsonSerializable(typeof(Stack<int>))]
        [JsonSerializable(typeof(StringStackWrapper))]
        [JsonSerializable(typeof(Queue<Queue<int>>))]
        [JsonSerializable(typeof(GenericQueueWrapper<StringQueueWrapper>))]
        [JsonSerializable(typeof(Queue<int[]>))]
        [JsonSerializable(typeof(Queue<int>[]), TypeInfoPropertyName = "ArrayOfIntQueue")]
        [JsonSerializable(typeof(Queue<int>))]
        [JsonSerializable(typeof(HashSet<HashSet<int>>))]
        [JsonSerializable(typeof(GenericHashSetWrapper<StringHashSetWrapper>))]
        [JsonSerializable(typeof(HashSet<int[]>))]
        [JsonSerializable(typeof(HashSet<int>[]), TypeInfoPropertyName = "ArrayOfIntHashSet")]
        [JsonSerializable(typeof(HashSet<int>))]
        [JsonSerializable(typeof(LinkedList<LinkedList<int>>))]
        [JsonSerializable(typeof(GenericLinkedListWrapper<StringLinkedListWrapper>))]
        [JsonSerializable(typeof(LinkedList<int[]>))]
        [JsonSerializable(typeof(LinkedList<int>[]), TypeInfoPropertyName = "ArrayOfIntLinkedList")]
        [JsonSerializable(typeof(LinkedList<int>))]
        [JsonSerializable(typeof(SortedSet<int>[]))]
        [JsonSerializable(typeof(StringSortedSetWrapper[]))]
        [JsonSerializable(typeof(SortedSet<int>))]
        [JsonSerializable(typeof(ClassWithGenericStructIListWrapper))]
        [JsonSerializable(typeof(ClassWithGenericStructICollectionWrapper))]
        [JsonSerializable(typeof(ClassWithGenericStructIDictionaryWrapper))]
        [JsonSerializable(typeof(ClassWithGenericStructISetWrapper))]
        [JsonSerializable(typeof(SimpleTestClassWithGenericStructCollectionWrappers))]
        [JsonSerializable(typeof(SimpleTestStructWithNullableGenericStructCollectionWrappers))]
        [JsonSerializable(typeof(SimpleTestClassWithGenericCollectionWrappers))]
        [JsonSerializable(typeof(string[]))]
        [JsonSerializable(typeof(List<string>))]
        [JsonSerializable(typeof(GenericListWrapper<string>))]
        [JsonSerializable(typeof(MyMyList<string>))]
        [JsonSerializable(typeof(MyListString))]
        [JsonSerializable(typeof(NetworkWrapper))]
        [JsonSerializable(typeof(Client))]
        [JsonSerializable(typeof(IReadOnlyDictionary<Uri, int>))]
        [JsonSerializable(typeof(IEnumerable<IEnumerable>))]
        [JsonSerializable(typeof(GenericIEnumerableWrapper<WrapperForIEnumerable>))]
        [JsonSerializable(typeof(IEnumerable))]
        [JsonSerializable(typeof(IEnumerable[]))]
        [JsonSerializable(typeof(IList<IList>))]
        [JsonSerializable(typeof(GenericIListWrapper<WrapperForIList>))]
        [JsonSerializable(typeof(IList))]
        [JsonSerializable(typeof(IList[]))]
        [JsonSerializable(typeof(StructWrapperForIList))]
        [JsonSerializable(typeof(StructWrapperForIList?))]
        [JsonSerializable(typeof(ClassWithStructIListWrapper))]
        [JsonSerializable(typeof(StructWrapperForIDictionary))]
        [JsonSerializable(typeof(StructWrapperForIDictionary?))]
        [JsonSerializable(typeof(ClassWithStructIDictionaryWrapper))]
        [JsonSerializable(typeof(WrapperForIList))]
        [JsonSerializable(typeof(ICollection<ICollection>))]
        [JsonSerializable(typeof(GenericICollectionWrapper<WrapperForICollection>))]
        [JsonSerializable(typeof(ICollection))]
        [JsonSerializable(typeof(ICollection[]))]
        [JsonSerializable(typeof(Stack<Stack>))]
        [JsonSerializable(typeof(Stack))]
        [JsonSerializable(typeof(Stack[]))]
        [JsonSerializable(typeof(Queue<Queue>))]
        [JsonSerializable(typeof(Queue))]
        [JsonSerializable(typeof(Queue[]))]
        [JsonSerializable(typeof(QueueWrapper))]
        [JsonSerializable(typeof(ArrayList))]
        [JsonSerializable(typeof(ArrayListWrapper))]
        [JsonSerializable(typeof(ArrayList[]))]
        [JsonSerializable(typeof(SimpleTestClassWithNonGenericCollectionWrappers))]
        [JsonSerializable(typeof(SimpleTestClassWithStructCollectionWrappers))]
        [JsonSerializable(typeof(SimpleTestStructWithNullableStructCollectionWrappers))]
        [JsonSerializable(typeof(Collection<bool>))]
        [JsonSerializable(typeof(ObservableCollection<bool>))]
        [JsonSerializable(typeof(SimpleKeyedCollection))]
        [JsonSerializable(typeof(ReadOnlyCollection<bool>))]
        [JsonSerializable(typeof(ReadOnlyObservableCollection<bool>))]
        [JsonSerializable(typeof(ReadOnlyDictionary<string, bool>))]
        [JsonSerializable(typeof(KeyedCollection<string, bool>))]
        [JsonSerializable(typeof(BitVector32))]
        [JsonSerializable(typeof(HybridDictionary))]
        [JsonSerializable(typeof(OrderedDictionary))]
        [JsonSerializable(typeof(ListDictionary))]
        [JsonSerializable(typeof(StringCollection))]
        [JsonSerializable(typeof(StringDictionary))]
        [JsonSerializable(typeof(IOrderedDictionary))]
        [JsonSerializable(typeof(NameValueCollection))]
        [JsonSerializable(typeof(ImmutableArray<ImmutableArray<int>>))]
        [JsonSerializable(typeof(ImmutableArray<int[]>))]
        [JsonSerializable(typeof(ImmutableArray<int>[]), TypeInfoPropertyName = "ArrayOfImmutableIntArray")]
        [JsonSerializable(typeof(ImmutableArray<int>))]
        [JsonSerializable(typeof(SimpleTestClassWithImmutableArray))]
        [JsonSerializable(typeof(IImmutableList<IImmutableList<int>>))]
        [JsonSerializable(typeof(IImmutableList<int[]>))]
        [JsonSerializable(typeof(IImmutableList<int>[]), TypeInfoPropertyName = "ArrayOfIImmutableIntArrayList")]
        [JsonSerializable(typeof(IImmutableList<int>))]
        [JsonSerializable(typeof(IImmutableStack<IImmutableStack<int>>))]
        [JsonSerializable(typeof(IImmutableStack<int[]>))]
        [JsonSerializable(typeof(IImmutableStack<int>[]), TypeInfoPropertyName = "ArrayOfIImmutableIntStack")]
        [JsonSerializable(typeof(IImmutableStack<int>))]
        [JsonSerializable(typeof(IImmutableQueue<IImmutableQueue<int>>))]
        [JsonSerializable(typeof(IImmutableQueue<int[]>))]
        [JsonSerializable(typeof(IImmutableQueue<int>[]), TypeInfoPropertyName = "ArrayOfIImmutableIntQueue")]
        [JsonSerializable(typeof(IImmutableQueue<int>))]
        [JsonSerializable(typeof(IImmutableSet<IImmutableSet<int>>))]
        [JsonSerializable(typeof(IImmutableSet<int[]>))]
        [JsonSerializable(typeof(IImmutableSet<int>[]), TypeInfoPropertyName = "ArrayOfIImmutableIntSet")]
        [JsonSerializable(typeof(IImmutableSet<int>))]
        [JsonSerializable(typeof(ImmutableHashSet<ImmutableHashSet<int>>))]
        [JsonSerializable(typeof(ImmutableHashSet<int[]>))]
        [JsonSerializable(typeof(ImmutableHashSet<int>[]), TypeInfoPropertyName = "ArrayOfImmutableIntHashSet")]
        [JsonSerializable(typeof(ImmutableHashSet<int>))]
        [JsonSerializable(typeof(ImmutableList<ImmutableList<int>>))]
        [JsonSerializable(typeof(ImmutableList<int[]>))]
        [JsonSerializable(typeof(ImmutableList<int>[]), TypeInfoPropertyName = "ArrayOfImmutableIntList")]
        [JsonSerializable(typeof(ImmutableList<int>))]
        [JsonSerializable(typeof(ImmutableStack<ImmutableStack<int>>))]
        [JsonSerializable(typeof(ImmutableStack<int[]>))]
        [JsonSerializable(typeof(ImmutableStack<int>[]), TypeInfoPropertyName = "ArrayOfImmutableIntStack")]
        [JsonSerializable(typeof(ImmutableStack<int>))]
        [JsonSerializable(typeof(ImmutableQueue<ImmutableQueue<int>>))]
        [JsonSerializable(typeof(ImmutableQueue<int[]>))]
        [JsonSerializable(typeof(ImmutableQueue<int>[]), TypeInfoPropertyName = "ArrayOfImmutableIntQueue")]
        [JsonSerializable(typeof(ImmutableQueue<int>))]
        [JsonSerializable(typeof(ImmutableSortedSet<int>[]))]
        [JsonSerializable(typeof(ImmutableSortedSet<int>))]
        [JsonSerializable(typeof(SimpleTestClassWithIImmutableDictionaryWrapper))]
        [JsonSerializable(typeof(SimpleTestClassWithImmutableListWrapper))]
        [JsonSerializable(typeof(SimpleTestClassWithImmutableStackWrapper))]
        [JsonSerializable(typeof(SimpleTestClassWithImmutableQueueWrapper))]
        [JsonSerializable(typeof(SimpleTestClassWithImmutableSetWrapper))]
        [JsonSerializable(typeof(ClassWithPopulatedDictionaryAndNoSetter))]
        [JsonSerializable(typeof(StringIImmutableQueueWrapper))]
        [JsonSerializable(typeof(StringIImmutableStackWrapper))]
        [JsonSerializable(typeof(ClassWithPopulatedDictionaryAndSetter))]
        [JsonSerializable(typeof(StringIImmutableListWrapper))]
        [JsonSerializable(typeof(SimpleTestClassWithObjectImmutableArray))]
        [JsonSerializable(typeof(ImmutableArray<string>))]
        [JsonSerializable(typeof(StringIImmutableSetWrapper))]
        [JsonSerializable(typeof(IEnumerable<ValueB>))]
        [JsonSerializable(typeof(ICollection<ICollection<int>>))]
        [JsonSerializable(typeof(SimpleTestClassWithStringIEnumerableWrapper))]
        [JsonSerializable(typeof(SimpleTestClassWithStringIReadOnlyCollectionWrapper))]
        [JsonSerializable(typeof(SimpleTestClassWithStringIReadOnlyListWrapper))]
        [JsonSerializable(typeof(SimpleTestClassWithStringToStringIReadOnlyDictionaryWrapper))]
        [JsonSerializable(typeof(Dictionary<string, int?>))]
        [JsonSerializable(typeof(GenericICollectionWrapper<GenericICollectionWrapper<string>>))]
        [JsonSerializable(typeof(GenericIEnumerableWrapperPrivateConstructor<string>))]
        [JsonSerializable(typeof(GenericIEnumerableWrapperInternalConstructor<string>))]
        [JsonSerializable(typeof(GenericICollectionWrapperPrivateConstructor<string>))]
        [JsonSerializable(typeof(GenericICollectionWrapperInternalConstructor<string>))]
        [JsonSerializable(typeof(GenericIListWrapperPrivateConstructor<string>))]
        [JsonSerializable(typeof(GenericIListWrapperInternalConstructor<string>))]
        [JsonSerializable(typeof(GenericISetWrapperPrivateConstructor<string>))]
        [JsonSerializable(typeof(GenericISetWrapperInternalConstructor<string>))]
        [JsonSerializable(typeof(GenericIDictionaryWrapperPrivateConstructor<string, string>))]
        [JsonSerializable(typeof(GenericIDictionaryWrapperInternalConstructor<string, string>))]
        [JsonSerializable(typeof(StringToStringIReadOnlyDictionaryWrapperPrivateConstructor))]
        [JsonSerializable(typeof(StringToStringIReadOnlyDictionaryWrapperInternalConstructor))]
        [JsonSerializable(typeof(GenericListWrapperPrivateConstructor<string>))]
        [JsonSerializable(typeof(GenericListWrapperInternalConstructor<string>))]
        [JsonSerializable(typeof(GenericQueueWrapperPrivateConstructor<string>))]
        [JsonSerializable(typeof(GenericQueueWrapperInternalConstructor<string>))]
        [JsonSerializable(typeof(GenericStackWrapperPrivateConstructor<string>))]
        [JsonSerializable(typeof(GenericStackWrapperInternalConstructor<string>))]
        [JsonSerializable(typeof(StringToGenericDictionaryWrapperPrivateConstructor<string>))]
        [JsonSerializable(typeof(StringToGenericDictionaryWrapperInternalConstructor<string>))]
        [JsonSerializable(typeof(SimpleTestClassWithNonGenericCollectionWrappers))]
        [JsonSerializable(typeof(SimpleTestClassWithIEnumerableWrapper))]
        [JsonSerializable(typeof(SimpleTestClassWithICollectionWrapper))]
        [JsonSerializable(typeof(SimpleTestClassWithStackWrapper))]
        [JsonSerializable(typeof(SimpleTestClassWithQueueWrapper))]
        [JsonSerializable(typeof(List<object>))]
        [JsonSerializable(typeof(WrapperForIEnumerable))]
        [JsonSerializable(typeof(GenericStackWrapper<StackWrapper>))]
        [JsonSerializable(typeof(DictionaryEntry))]
        [JsonSerializable(typeof(Dictionary<string, string>[]))]
        [JsonSerializable(typeof(GenericQueueWrapper<QueueWrapper>))]
        [JsonSerializable(typeof(WrapperForIEnumerablePrivateConstructor))]
        [JsonSerializable(typeof(WrapperForIEnumerableInternalConstructor))]
        [JsonSerializable(typeof(WrapperForICollectionPrivateConstructor))]
        [JsonSerializable(typeof(WrapperForICollectionInternalConstructor))]
        [JsonSerializable(typeof(WrapperForIListPrivateConstructor))]
        [JsonSerializable(typeof(WrapperForIListInternalConstructor))]
        [JsonSerializable(typeof(WrapperForIDictionaryPrivateConstructor))]
        [JsonSerializable(typeof(WrapperForIDictionaryInternalConstructor))]
        [JsonSerializable(typeof(IDerivedICollectionOfT<string>))]
        [JsonSerializable(typeof(IDerivedIList))]
        [JsonSerializable(typeof(IDerivedISetOfT<string>))]
        [JsonSerializable(typeof(ReadOnlyWrapperForIList))]
        [JsonSerializable(typeof(ReadOnlyStringIListWrapper))]
        [JsonSerializable(typeof(ReadOnlyStringICollectionWrapper))]
        [JsonSerializable(typeof(ReadOnlyStringISetWrapper))]
        [JsonSerializable(typeof(ReadOnlyWrapperForIDictionary))]
        [JsonSerializable(typeof(ReadOnlyStringToStringIDictionaryWrapper))]
        [JsonSerializable(typeof(Dictionary<string, int?>[]))]
        [JsonSerializable(typeof(KeyValuePair<string, int>))]
        [JsonSerializable(typeof(List<KeyValuePair<string, int>>))]
        [JsonSerializable(typeof(KeyValuePair<string, List<int>>))]
        [JsonSerializable(typeof(KeyValuePair<string, KeyValuePair<int, int>>))]
        [JsonSerializable(typeof(KeyValuePair<string, string>))]
        [JsonSerializable(typeof(KeyValuePair<string, object>))]
        [JsonSerializable(typeof(KeyValuePair<string, SimpleClassWithKeyValuePairs>))]
        [JsonSerializable(typeof(KeyValuePair<string, KeyValuePair<string, string>>))]
        [JsonSerializable(typeof(KeyValuePair<string, KeyValuePair<string, object>>))]
        [JsonSerializable(typeof(KeyValuePair<string, KeyValuePair<string, SimpleClassWithKeyValuePairs>>))]
        [JsonSerializable(typeof(SimpleClassWithKeyValuePairs))]
        [JsonSerializable(typeof(KeyNameNullPolicy))]
        [JsonSerializable(typeof(ValueNameNullPolicy))]
        [JsonSerializable(typeof(KeyValuePair<int, int>))]
        [JsonSerializable(typeof(KeyValuePair<string, KeyValuePair<string, int>>))]
        [JsonSerializable(typeof(StackWrapper))]
        internal sealed partial class CollectionTestsContext_Metadata : JsonSerializerContext
        {
        }
    }

    public partial class CollectionTests_Default : CollectionTests_Metadata
    {
        public CollectionTests_Default()
            : base(new StringSerializerWrapper(CollectionTestsContext_Default.Default, (options) => new CollectionTestsContext_Default(options)))
        {
        }

        [JsonSerializable(typeof(ConcurrentDictionary<string, string>))]
        [JsonSerializable(typeof(ConcurrentQueue<string>))]
        [JsonSerializable(typeof(ConcurrentStack<string>))]
        [JsonSerializable(typeof(BlockingCollection<string>))]
        [JsonSerializable(typeof(ConcurrentBag<string>))]
        [JsonSerializable(typeof(GenericConcurrentQueuePrivateConstructor<string>))]
        [JsonSerializable(typeof(GenericConcurrentQueueInternalConstructor<string>))]
        [JsonSerializable(typeof(GenericConcurrentStackPrivateConstructor<string>))]
        [JsonSerializable(typeof(GenericConcurrentStackInternalConstructor<string>))]
        [JsonSerializable(typeof(IDictionary))]
        [JsonSerializable(typeof(Dictionary<string, string>))]
        [JsonSerializable(typeof(SortedDictionary<string, string>))]
        [JsonSerializable(typeof(object))]
        [JsonSerializable(typeof(IDictionary<string, string>))]
        [JsonSerializable(typeof(IReadOnlyDictionary<string, string>))]
        [JsonSerializable(typeof(ImmutableDictionary<string, string>))]
        [JsonSerializable(typeof(IImmutableDictionary<string, string>))]
        [JsonSerializable(typeof(ImmutableSortedDictionary<string, string>))]
        [JsonSerializable(typeof(Hashtable))]
        [JsonSerializable(typeof(SortedList))]
        [JsonSerializable(typeof(WrapperForIDictionary))]
        [JsonSerializable(typeof(StringToStringDictionaryWrapper))]
        [JsonSerializable(typeof(StringToStringSortedDictionaryWrapper))]
        [JsonSerializable(typeof(GenericIDictionaryWrapper<string, string>))]
        [JsonSerializable(typeof(GenericIReadOnlyDictionaryWrapper<string, string>))]
        [JsonSerializable(typeof(StringToStringIImmutableDictionaryWrapper))]
        [JsonSerializable(typeof(HashtableWrapper))]
        [JsonSerializable(typeof(SortedListWrapper))]
        [JsonSerializable(typeof(GenericStructIDictionaryWrapper<string, string>))]
        [JsonSerializable(typeof(GenericStructIDictionaryWrapper<string, string>?))]
        [JsonSerializable(typeof(StructWrapperForIDictionary))]
        [JsonSerializable(typeof(Dictionary<string, object>))]
        [JsonSerializable(typeof(IDictionary<string, object>))]
        [JsonSerializable(typeof(GenericIDictionaryWrapper<string, object>))]
        [JsonSerializable(typeof(GenericIDictionaryWrapper<string, string>))]
        [JsonSerializable(typeof(PocoDictionary))]
        [JsonSerializable(typeof(IDictionary<string, List<int>>))]
        [JsonSerializable(typeof(ImmutableDictionary<string, List<int>>))]
        [JsonSerializable(typeof(IImmutableDictionary<string, List<int>>))]
        [JsonSerializable(typeof(Dictionary<string, int[]>))]
        [JsonSerializable(typeof(List<Dictionary<string, int>>))]
        [JsonSerializable(typeof(List<ImmutableSortedDictionary<string, int>>))]
        [JsonSerializable(typeof(Dictionary<string, int>[]), TypeInfoPropertyName = "ArrayOfDictionaryStringInt")]
        [JsonSerializable(typeof(ImmutableSortedDictionary<string, int>[]))]
        [JsonSerializable(typeof(Dictionary<string, Dictionary<string, int>>))]
        [JsonSerializable(typeof(ImmutableSortedDictionary<string, ImmutableSortedDictionary<string, int>>))]
        [JsonSerializable(typeof(Dictionary<string, Dictionary<string, Dictionary<string, int>>>))]
        [JsonSerializable(typeof(Dictionary<string, Dictionary<string, int>[]>))]
        [JsonSerializable(typeof(SimpleTestClass))]
        [JsonSerializable(typeof(Dictionary<string, int>))]
        [JsonSerializable(typeof(Dictionary<string, JsonElement>))]
        [JsonSerializable(typeof(PocoDuplicate))]
        [JsonSerializable(typeof(ClassWithIgnoredDictionary1))]
        [JsonSerializable(typeof(ClassWithIgnoredDictionary2))]
        [JsonSerializable(typeof(ClassWithIgnoredDictionary3))]
        [JsonSerializable(typeof(ClassWithIgnoredDictionary4))]
        [JsonSerializable(typeof(ClassWithIgnoredDictionary5))]
        [JsonSerializable(typeof(ClassWithIgnoredDictionary6))]
        [JsonSerializable(typeof(ClassWithIgnoredDictionary7))]
        [JsonSerializable(typeof(ClassWithIgnoredIDictionary))]
        [JsonSerializable(typeof(ClassWithIgnoreAttributeDictionary))]
        [JsonSerializable(typeof(ClassWithIgnoredImmutableDictionary))]
        [JsonSerializable(typeof(Dictionary<string, AllSingleUpperProperties_Child>))]
        [JsonSerializable(typeof(AllSingleUpperPropertiesParent))]
        [JsonSerializable(typeof(ClassWithDictionaryOfString_ChildWithDictionaryOfString))]
        [JsonSerializable(typeof(ClassWithDictionaryOfString_ChildWithDictionaryOfString))]
        [JsonSerializable(typeof(ClassWithDictionaryAndProperty_DictionaryFirst))]
        [JsonSerializable(typeof(ClassWithDictionaryAndProperty_DictionaryLast))]
        [JsonSerializable(typeof(SimpleClassWithDictionaries))]
        [JsonSerializable(typeof(DictionaryThatOnlyImplementsIDictionaryOfStringTValue<int>))]
        [JsonSerializable(typeof(DictionaryThatOnlyImplementsIDictionaryOfStringTValue<Poco>))]
        [JsonSerializable(typeof(DictionaryThatOnlyImplementsIDictionaryOfStringPoco))]
        [JsonSerializable(typeof(DictionaryThatHasIncompatibleEnumerator))]
        [JsonSerializable(typeof(Dictionary<string, ClassWithoutParameterlessCtor>))]
        [JsonSerializable(typeof(StructWrapperForIDictionary?))]
        [JsonSerializable(typeof(ClassWithStructIDictionaryWrapper))]
        [JsonSerializable(typeof(Poco))]
        [JsonSerializable(typeof(JsonElement))]
        [JsonSerializable(typeof(string))]
        [JsonSerializable(typeof(IDictionary<int, int>))]
        [JsonSerializable(typeof(Dictionary<string, ClassWithInternalParameterlessConstructor>))]
        [JsonSerializable(typeof(Dictionary<string, ClassWithPrivateParameterlessConstructor>))]
        [JsonSerializable(typeof(Dictionary<string, Dictionary<string, CustomClass>>))]
        [JsonSerializable(typeof(TestClassWithDictionary))]
        [JsonSerializable(typeof(IReadOnlyDictionary<Uri, int>))]
        [JsonSerializable(typeof(GenericIReadOnlyDictionaryWrapper<Uri, int>))]
        [JsonSerializable(typeof(List<int>))]
        [JsonSerializable(typeof(IReadOnlyDictionary<int, int>))]
        [JsonSerializable(typeof(Dictionary<string, CustomClass>))]
        [JsonSerializable(typeof(IDerivedIDictionaryOfTKeyTValue<string, string>))]
        [JsonSerializable(typeof(List<List<int>>))]
        [JsonSerializable(typeof(GenericListWrapper<StringListWrapper>))]
        [JsonSerializable(typeof(List<List<int>>))]
        [JsonSerializable(typeof(GenericListWrapper<StringListWrapper>))]
        [JsonSerializable(typeof(List<int[]>))]
        [JsonSerializable(typeof(GenericListWrapper<string[]>))]
        [JsonSerializable(typeof(List<int>[]), TypeInfoPropertyName = "ArrayOfIntList")]
        [JsonSerializable(typeof(StringListWrapper[]))]
        [JsonSerializable(typeof(List<int>))]
        [JsonSerializable(typeof(StringListWrapper))]
        [JsonSerializable(typeof(IEnumerable<IEnumerable<int>>))]
        [JsonSerializable(typeof(GenericIEnumerableWrapper<StringIEnumerableWrapper>))]
        [JsonSerializable(typeof(IEnumerable<int[]>))]
        [JsonSerializable(typeof(GenericIEnumerableWrapper<int[]>))]
        [JsonSerializable(typeof(IEnumerable<int>[]), TypeInfoPropertyName = "ArrayOfIntIEnumerable")]
        [JsonSerializable(typeof(StringIEnumerableWrapper[]))]
        [JsonSerializable(typeof(IEnumerable<int>))]
        [JsonSerializable(typeof(StringIEnumerableWrapper))]
        [JsonSerializable(typeof(IList<IList<int>>))]
        [JsonSerializable(typeof(GenericIListWrapper<StringIListWrapper>))]
        [JsonSerializable(typeof(IList<int[]>))]
        [JsonSerializable(typeof(GenericIListWrapper<string[]>))]
        [JsonSerializable(typeof(IList<int>[]), TypeInfoPropertyName = "ArrayOfIntIList")]
        [JsonSerializable(typeof(StringIListWrapper[]))]
        [JsonSerializable(typeof(IList<int>))]
        [JsonSerializable(typeof(StringIListWrapper))]
        [JsonSerializable(typeof(GenericStructIListWrapper<int>))]
        [JsonSerializable(typeof(GenericStructIListWrapper<int>?))]
        [JsonSerializable(typeof(GenericStructICollectionWrapper<int>))]
        [JsonSerializable(typeof(GenericStructICollectionWrapper<int>?))]
        [JsonSerializable(typeof(ICollection<ICollection<int>>))]
        [JsonSerializable(typeof(ICollection<int[]>))]
        [JsonSerializable(typeof(GenericICollectionWrapper<string[]>))]
        [JsonSerializable(typeof(ICollection<int>[]), TypeInfoPropertyName = "ArrayOfIntICollection")]
        [JsonSerializable(typeof(ICollection<int>))]
        [JsonSerializable(typeof(GenericICollectionWrapper<string>))]
        [JsonSerializable(typeof(IReadOnlyCollection<IReadOnlyCollection<int>>))]
        [JsonSerializable(typeof(GenericIReadOnlyCollectionWrapper<WrapperForIReadOnlyCollectionOfT<string>>))]
        [JsonSerializable(typeof(IReadOnlyCollection<int[]>))]
        [JsonSerializable(typeof(GenericIReadOnlyCollectionWrapper<int[]>))]
        [JsonSerializable(typeof(IReadOnlyCollection<int>[]), TypeInfoPropertyName = "ArrayOfIntIReadOnlyCollection")]
        [JsonSerializable(typeof(WrapperForIReadOnlyCollectionOfT<string>[]))]
        [JsonSerializable(typeof(IReadOnlyCollection<int>))]
        [JsonSerializable(typeof(WrapperForIReadOnlyCollectionOfT<string>))]
        [JsonSerializable(typeof(IReadOnlyList<IReadOnlyList<int>>))]
        [JsonSerializable(typeof(GenericIReadOnlyListWrapper<StringIReadOnlyListWrapper>))]
        [JsonSerializable(typeof(IReadOnlyList<int[]>))]
        [JsonSerializable(typeof(GenericIReadOnlyListWrapper<string[]>))]
        [JsonSerializable(typeof(IReadOnlyList<int>[]), TypeInfoPropertyName = "ArrayOfIntIReadOnlyList")]
        [JsonSerializable(typeof(StringIReadOnlyListWrapper[]))]
        [JsonSerializable(typeof(IReadOnlyList<int>))]
        [JsonSerializable(typeof(StringIReadOnlyListWrapper))]
        [JsonSerializable(typeof(ISet<ISet<int>>))]
        [JsonSerializable(typeof(GenericISetWrapper<StringISetWrapper>))]
        [JsonSerializable(typeof(GenericStructISetWrapper<int>))]
        [JsonSerializable(typeof(GenericStructISetWrapper<int>?))]
        [JsonSerializable(typeof(ISet<HashSet<int>>))]
        [JsonSerializable(typeof(HashSet<ISet<int>>))]
        [JsonSerializable(typeof(ISet<int[]>))]
        [JsonSerializable(typeof(ISet<int>[]), TypeInfoPropertyName = "ArrayOfIntISet")]
        [JsonSerializable(typeof(ISet<int>))]
        [JsonSerializable(typeof(Stack<Stack<int>>))]
        [JsonSerializable(typeof(GenericStackWrapper<StringStackWrapper>))]
        [JsonSerializable(typeof(Stack<int[]>))]
        [JsonSerializable(typeof(GenericStackWrapper<string[]>))]
        [JsonSerializable(typeof(Stack<int>[]), TypeInfoPropertyName = "ArrayOfIntStack")]
        [JsonSerializable(typeof(StringStackWrapper[]))]
        [JsonSerializable(typeof(Stack<int>))]
        [JsonSerializable(typeof(StringStackWrapper))]
        [JsonSerializable(typeof(Queue<Queue<int>>))]
        [JsonSerializable(typeof(GenericQueueWrapper<StringQueueWrapper>))]
        [JsonSerializable(typeof(Queue<int[]>))]
        [JsonSerializable(typeof(Queue<int>[]), TypeInfoPropertyName = "ArrayOfIntQueue")]
        [JsonSerializable(typeof(Queue<int>))]
        [JsonSerializable(typeof(HashSet<HashSet<int>>))]
        [JsonSerializable(typeof(GenericHashSetWrapper<StringHashSetWrapper>))]
        [JsonSerializable(typeof(HashSet<int[]>))]
        [JsonSerializable(typeof(HashSet<int>[]), TypeInfoPropertyName = "ArrayOfIntHashSet")]
        [JsonSerializable(typeof(HashSet<int>))]
        [JsonSerializable(typeof(LinkedList<LinkedList<int>>))]
        [JsonSerializable(typeof(GenericLinkedListWrapper<StringLinkedListWrapper>))]
        [JsonSerializable(typeof(LinkedList<int[]>))]
        [JsonSerializable(typeof(LinkedList<int>[]), TypeInfoPropertyName = "ArrayOfIntLinkedList")]
        [JsonSerializable(typeof(LinkedList<int>))]
        [JsonSerializable(typeof(SortedSet<int>[]))]
        [JsonSerializable(typeof(StringSortedSetWrapper[]))]
        [JsonSerializable(typeof(SortedSet<int>))]
        [JsonSerializable(typeof(ClassWithGenericStructIListWrapper))]
        [JsonSerializable(typeof(ClassWithGenericStructICollectionWrapper))]
        [JsonSerializable(typeof(ClassWithGenericStructIDictionaryWrapper))]
        [JsonSerializable(typeof(ClassWithGenericStructISetWrapper))]
        [JsonSerializable(typeof(SimpleTestClassWithGenericStructCollectionWrappers))]
        [JsonSerializable(typeof(SimpleTestStructWithNullableGenericStructCollectionWrappers))]
        [JsonSerializable(typeof(SimpleTestClassWithGenericCollectionWrappers))]
        [JsonSerializable(typeof(string[]))]
        [JsonSerializable(typeof(List<string>))]
        [JsonSerializable(typeof(GenericListWrapper<string>))]
        [JsonSerializable(typeof(MyMyList<string>))]
        [JsonSerializable(typeof(MyListString))]
        [JsonSerializable(typeof(NetworkWrapper))]
        [JsonSerializable(typeof(Client))]
        [JsonSerializable(typeof(IReadOnlyDictionary<Uri, int>))]
        [JsonSerializable(typeof(IEnumerable<IEnumerable>))]
        [JsonSerializable(typeof(GenericIEnumerableWrapper<WrapperForIEnumerable>))]
        [JsonSerializable(typeof(IEnumerable))]
        [JsonSerializable(typeof(IEnumerable[]))]
        [JsonSerializable(typeof(IList<IList>))]
        [JsonSerializable(typeof(GenericIListWrapper<WrapperForIList>))]
        [JsonSerializable(typeof(IList))]
        [JsonSerializable(typeof(IList[]))]
        [JsonSerializable(typeof(StructWrapperForIList))]
        [JsonSerializable(typeof(StructWrapperForIList?))]
        [JsonSerializable(typeof(ClassWithStructIListWrapper))]
        [JsonSerializable(typeof(StructWrapperForIDictionary))]
        [JsonSerializable(typeof(StructWrapperForIDictionary?))]
        [JsonSerializable(typeof(ClassWithStructIDictionaryWrapper))]
        [JsonSerializable(typeof(WrapperForIList))]
        [JsonSerializable(typeof(ICollection<ICollection>))]
        [JsonSerializable(typeof(GenericICollectionWrapper<WrapperForICollection>))]
        [JsonSerializable(typeof(ICollection))]
        [JsonSerializable(typeof(ICollection[]))]
        [JsonSerializable(typeof(Stack<Stack>))]
        [JsonSerializable(typeof(Stack))]
        [JsonSerializable(typeof(Stack[]))]
        [JsonSerializable(typeof(Queue<Queue>))]
        [JsonSerializable(typeof(Queue))]
        [JsonSerializable(typeof(Queue[]))]
        [JsonSerializable(typeof(QueueWrapper))]
        [JsonSerializable(typeof(ArrayList))]
        [JsonSerializable(typeof(ArrayListWrapper))]
        [JsonSerializable(typeof(ArrayList[]))]
        [JsonSerializable(typeof(SimpleTestClassWithNonGenericCollectionWrappers))]
        [JsonSerializable(typeof(SimpleTestClassWithStructCollectionWrappers))]
        [JsonSerializable(typeof(SimpleTestStructWithNullableStructCollectionWrappers))]
        [JsonSerializable(typeof(Collection<bool>))]
        [JsonSerializable(typeof(ObservableCollection<bool>))]
        [JsonSerializable(typeof(SimpleKeyedCollection))]
        [JsonSerializable(typeof(ReadOnlyCollection<bool>))]
        [JsonSerializable(typeof(ReadOnlyObservableCollection<bool>))]
        [JsonSerializable(typeof(ReadOnlyDictionary<string, bool>))]
        [JsonSerializable(typeof(KeyedCollection<string, bool>))]
        [JsonSerializable(typeof(BitVector32))]
        [JsonSerializable(typeof(HybridDictionary))]
        [JsonSerializable(typeof(OrderedDictionary))]
        [JsonSerializable(typeof(ListDictionary))]
        [JsonSerializable(typeof(StringCollection))]
        [JsonSerializable(typeof(StringDictionary))]
        [JsonSerializable(typeof(IOrderedDictionary))]
        [JsonSerializable(typeof(NameValueCollection))]
        [JsonSerializable(typeof(ImmutableArray<ImmutableArray<int>>))]
        [JsonSerializable(typeof(ImmutableArray<int[]>))]
        [JsonSerializable(typeof(ImmutableArray<int>[]), TypeInfoPropertyName = "ArrayOfImmutableIntArray")]
        [JsonSerializable(typeof(ImmutableArray<int>))]
        [JsonSerializable(typeof(SimpleTestClassWithImmutableArray))]
        [JsonSerializable(typeof(IImmutableList<IImmutableList<int>>))]
        [JsonSerializable(typeof(IImmutableList<int[]>))]
        [JsonSerializable(typeof(IImmutableList<int>[]), TypeInfoPropertyName = "ArrayOfIImmutableIntArrayList")]
        [JsonSerializable(typeof(IImmutableList<int>))]
        [JsonSerializable(typeof(IImmutableStack<IImmutableStack<int>>))]
        [JsonSerializable(typeof(IImmutableStack<int[]>))]
        [JsonSerializable(typeof(IImmutableStack<int>[]), TypeInfoPropertyName = "ArrayOfIImmutableIntStack")]
        [JsonSerializable(typeof(IImmutableStack<int>))]
        [JsonSerializable(typeof(IImmutableQueue<IImmutableQueue<int>>))]
        [JsonSerializable(typeof(IImmutableQueue<int[]>))]
        [JsonSerializable(typeof(IImmutableQueue<int>[]), TypeInfoPropertyName = "ArrayOfIImmutableIntQueue")]
        [JsonSerializable(typeof(IImmutableQueue<int>))]
        [JsonSerializable(typeof(IImmutableSet<IImmutableSet<int>>))]
        [JsonSerializable(typeof(IImmutableSet<int[]>))]
        [JsonSerializable(typeof(IImmutableSet<int>[]), TypeInfoPropertyName = "ArrayOfIImmutableIntSet")]
        [JsonSerializable(typeof(IImmutableSet<int>))]
        [JsonSerializable(typeof(ImmutableHashSet<ImmutableHashSet<int>>))]
        [JsonSerializable(typeof(ImmutableHashSet<int[]>))]
        [JsonSerializable(typeof(ImmutableHashSet<int>[]), TypeInfoPropertyName = "ArrayOfImmutableIntHashSet")]
        [JsonSerializable(typeof(ImmutableHashSet<int>))]
        [JsonSerializable(typeof(ImmutableList<ImmutableList<int>>))]
        [JsonSerializable(typeof(ImmutableList<int[]>))]
        [JsonSerializable(typeof(ImmutableList<int>[]), TypeInfoPropertyName = "ArrayOfImmutableIntList")]
        [JsonSerializable(typeof(ImmutableList<int>))]
        [JsonSerializable(typeof(ImmutableStack<ImmutableStack<int>>))]
        [JsonSerializable(typeof(ImmutableStack<int[]>))]
        [JsonSerializable(typeof(ImmutableStack<int>[]), TypeInfoPropertyName = "ArrayOfImmutableIntStack")]
        [JsonSerializable(typeof(ImmutableStack<int>))]
        [JsonSerializable(typeof(ImmutableQueue<ImmutableQueue<int>>))]
        [JsonSerializable(typeof(ImmutableQueue<int[]>))]
        [JsonSerializable(typeof(ImmutableQueue<int>[]), TypeInfoPropertyName = "ArrayOfImmutableIntQueue")]
        [JsonSerializable(typeof(ImmutableQueue<int>))]
        [JsonSerializable(typeof(ImmutableSortedSet<int>[]))]
        [JsonSerializable(typeof(ImmutableSortedSet<int>))]
        [JsonSerializable(typeof(SimpleTestClassWithIImmutableDictionaryWrapper))]
        [JsonSerializable(typeof(SimpleTestClassWithImmutableListWrapper))]
        [JsonSerializable(typeof(SimpleTestClassWithImmutableStackWrapper))]
        [JsonSerializable(typeof(SimpleTestClassWithImmutableQueueWrapper))]
        [JsonSerializable(typeof(SimpleTestClassWithImmutableSetWrapper))]
        [JsonSerializable(typeof(ClassWithPopulatedDictionaryAndNoSetter))]
        [JsonSerializable(typeof(StringIImmutableQueueWrapper))]
        [JsonSerializable(typeof(StringIImmutableStackWrapper))]
        [JsonSerializable(typeof(ClassWithPopulatedDictionaryAndSetter))]
        [JsonSerializable(typeof(StringIImmutableListWrapper))]
        [JsonSerializable(typeof(SimpleTestClassWithObjectImmutableArray))]
        [JsonSerializable(typeof(ImmutableArray<string>))]
        [JsonSerializable(typeof(StringIImmutableSetWrapper))]
        [JsonSerializable(typeof(IEnumerable<ValueB>))]
        [JsonSerializable(typeof(ICollection<ICollection<int>>))]
        [JsonSerializable(typeof(SimpleTestClassWithStringIEnumerableWrapper))]
        [JsonSerializable(typeof(SimpleTestClassWithStringIReadOnlyCollectionWrapper))]
        [JsonSerializable(typeof(SimpleTestClassWithStringIReadOnlyListWrapper))]
        [JsonSerializable(typeof(SimpleTestClassWithStringToStringIReadOnlyDictionaryWrapper))]
        [JsonSerializable(typeof(Dictionary<string, int?>))]
        [JsonSerializable(typeof(GenericICollectionWrapper<GenericICollectionWrapper<string>>))]
        [JsonSerializable(typeof(GenericIEnumerableWrapperPrivateConstructor<string>))]
        [JsonSerializable(typeof(GenericIEnumerableWrapperInternalConstructor<string>))]
        [JsonSerializable(typeof(GenericICollectionWrapperPrivateConstructor<string>))]
        [JsonSerializable(typeof(GenericICollectionWrapperInternalConstructor<string>))]
        [JsonSerializable(typeof(GenericIListWrapperPrivateConstructor<string>))]
        [JsonSerializable(typeof(GenericIListWrapperInternalConstructor<string>))]
        [JsonSerializable(typeof(GenericISetWrapperPrivateConstructor<string>))]
        [JsonSerializable(typeof(GenericISetWrapperInternalConstructor<string>))]
        [JsonSerializable(typeof(GenericIDictionaryWrapperPrivateConstructor<string, string>))]
        [JsonSerializable(typeof(GenericIDictionaryWrapperInternalConstructor<string, string>))]
        [JsonSerializable(typeof(StringToStringIReadOnlyDictionaryWrapperPrivateConstructor))]
        [JsonSerializable(typeof(StringToStringIReadOnlyDictionaryWrapperInternalConstructor))]
        [JsonSerializable(typeof(GenericListWrapperPrivateConstructor<string>))]
        [JsonSerializable(typeof(GenericListWrapperInternalConstructor<string>))]
        [JsonSerializable(typeof(GenericQueueWrapperPrivateConstructor<string>))]
        [JsonSerializable(typeof(GenericQueueWrapperInternalConstructor<string>))]
        [JsonSerializable(typeof(GenericStackWrapperPrivateConstructor<string>))]
        [JsonSerializable(typeof(GenericStackWrapperInternalConstructor<string>))]
        [JsonSerializable(typeof(StringToGenericDictionaryWrapperPrivateConstructor<string>))]
        [JsonSerializable(typeof(StringToGenericDictionaryWrapperInternalConstructor<string>))]
        [JsonSerializable(typeof(SimpleTestClassWithNonGenericCollectionWrappers))]
        [JsonSerializable(typeof(SimpleTestClassWithIEnumerableWrapper))]
        [JsonSerializable(typeof(SimpleTestClassWithICollectionWrapper))]
        [JsonSerializable(typeof(SimpleTestClassWithStackWrapper))]
        [JsonSerializable(typeof(SimpleTestClassWithQueueWrapper))]
        [JsonSerializable(typeof(List<object>))]
        [JsonSerializable(typeof(WrapperForIEnumerable))]
        [JsonSerializable(typeof(GenericStackWrapper<StackWrapper>))]
        [JsonSerializable(typeof(DictionaryEntry))]
        [JsonSerializable(typeof(Dictionary<string, string>[]))]
        [JsonSerializable(typeof(GenericQueueWrapper<QueueWrapper>))]
        [JsonSerializable(typeof(WrapperForIEnumerablePrivateConstructor))]
        [JsonSerializable(typeof(WrapperForIEnumerableInternalConstructor))]
        [JsonSerializable(typeof(WrapperForICollectionPrivateConstructor))]
        [JsonSerializable(typeof(WrapperForICollectionInternalConstructor))]
        [JsonSerializable(typeof(WrapperForIListPrivateConstructor))]
        [JsonSerializable(typeof(WrapperForIListInternalConstructor))]
        [JsonSerializable(typeof(WrapperForIDictionaryPrivateConstructor))]
        [JsonSerializable(typeof(WrapperForIDictionaryInternalConstructor))]
        [JsonSerializable(typeof(IDerivedICollectionOfT<string>))]
        [JsonSerializable(typeof(IDerivedIList))]
        [JsonSerializable(typeof(IDerivedISetOfT<string>))]
        [JsonSerializable(typeof(ReadOnlyWrapperForIList))]
        [JsonSerializable(typeof(ReadOnlyStringIListWrapper))]
        [JsonSerializable(typeof(ReadOnlyStringICollectionWrapper))]
        [JsonSerializable(typeof(ReadOnlyStringISetWrapper))]
        [JsonSerializable(typeof(ReadOnlyWrapperForIDictionary))]
        [JsonSerializable(typeof(ReadOnlyStringToStringIDictionaryWrapper))]
        [JsonSerializable(typeof(Dictionary<string, int?>[]))]
        [JsonSerializable(typeof(KeyValuePair<string, int>))]
        [JsonSerializable(typeof(List<KeyValuePair<string, int>>))]
        [JsonSerializable(typeof(KeyValuePair<string, List<int>>))]
        [JsonSerializable(typeof(KeyValuePair<string, KeyValuePair<int, int>>))]
        [JsonSerializable(typeof(KeyValuePair<string, string>))]
        [JsonSerializable(typeof(KeyValuePair<string, object>))]
        [JsonSerializable(typeof(KeyValuePair<string, SimpleClassWithKeyValuePairs>))]
        [JsonSerializable(typeof(KeyValuePair<string, KeyValuePair<string, string>>))]
        [JsonSerializable(typeof(KeyValuePair<string, KeyValuePair<string, object>>))]
        [JsonSerializable(typeof(KeyValuePair<string, KeyValuePair<string, SimpleClassWithKeyValuePairs>>))]
        [JsonSerializable(typeof(SimpleClassWithKeyValuePairs))]
        [JsonSerializable(typeof(KeyNameNullPolicy))]
        [JsonSerializable(typeof(ValueNameNullPolicy))]
        [JsonSerializable(typeof(KeyValuePair<int, int>))]
        [JsonSerializable(typeof(KeyValuePair<string, KeyValuePair<string, int>>))]
        [JsonSerializable(typeof(StackWrapper))]
        internal sealed partial class CollectionTestsContext_Default : JsonSerializerContext
        {
        }
    }
}
