/*
 *   This file is part of Dianara
 *   Copyright 2012-2014  JanKusanagi <janjabber@gmail.com>
 *
 *   This program is free software; you can redistribute it and/or modify
 *   it under the terms of the GNU General Public License as published by
 *   the Free Software Foundation; either version 2 of the License, or
 *   (at your option) any later version.
 *
 *   This program is distributed in the hope that it will be useful,
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *   GNU General Public License for more details.
 *
 *   You should have received a copy of the GNU General Public License
 *   along with this program; if not, write to the
 *   Free Software Foundation, Inc.,
 *   51 Franklin Street, Fifth Floor, Boston, MA  02110-1301  USA .
 */

#include "minorfeeditem.h"


MinorFeedItem::MinorFeedItem(ASActivity *activity,
                             QString activityActorAvatarFilename,
                             PumpController *pumpController,
                             QWidget *parent) : QFrame(parent)
{
    this->pController = pumpController;

    this->itemIsNew = false;

    // This sizePolicy prevents cut messages, and the huge space at the end
    // of the feed, after clicking "More" several times
    this->setSizePolicy(QSizePolicy::Minimum,
                        QSizePolicy::Minimum);

    activity->setParent(this); // reparent the passed activity

    QFont mainFont;
    mainFont.setPointSize(mainFont.pointSize() - 2);

    QString authorId = activity->getAuthorId();

    avatarLabel = new QLabel();
    avatarLabel->setToolTip(authorId);
    if (QFile::exists(activityActorAvatarFilename))
    {
        avatarLabel->setPixmap(QPixmap(activityActorAvatarFilename)
                               .scaledToWidth(32, Qt::SmoothTransformation));
    }
    else
    {
        avatarLabel->setPixmap(QIcon::fromTheme("user-identity")
                               .pixmap(32,32)
                               .scaledToWidth(32, Qt::SmoothTransformation));
    }

    QString generator = activity->getGenerator();
    if (!generator.isEmpty())
    {
        generator = "\n"
                  + tr("using %1",
                       "Application used to generate this activity").arg(generator);
    }

    QString timestamp = activity->getCreatedAt();
    timestampLabel = new QLabel(Timestamp::fuzzyTime(timestamp));
    mainFont.setBold(true);
    timestampLabel->setFont(mainFont);
    timestampLabel->setWordWrap(true);
    timestampLabel->setToolTip(Timestamp::localTimeDate(timestamp) + generator);
    timestampLabel->setAlignment(Qt::AlignCenter);
    timestampLabel->setAutoFillBackground(true);
    timestampLabel->setForegroundRole(QPalette::Text);
    timestampLabel->setBackgroundRole(QPalette::Base);
    timestampLabel->setFrameStyle(QFrame::Panel | QFrame::Raised);


    QString activityTooltip;
    QString activityAuthorName = activity->object()->getAuthorName();
    if (activityAuthorName.isEmpty())
    {
        activityAuthorName = activity->object()->getAuthorId();
    }

    // If there's a name or an ID, show it
    if (!activityAuthorName.isEmpty())
    {
        activityTooltip = "[ " + activityAuthorName + " ]"
                          "<br><br>";
    }

    QString activityObjectContent = activity->object()->getContent();
    if (!activityObjectContent.isEmpty())
    {
        activityTooltip.append("<b></b>");
        activityTooltip.append(activityObjectContent);
    }

    activityDescriptionLabel = new QLabel(activity->getContent());
    mainFont.setBold(false);
    activityDescriptionLabel->setFont(mainFont);
    activityDescriptionLabel->setWordWrap(true);
    activityDescriptionLabel->setOpenExternalLinks(true);
    activityDescriptionLabel->setToolTip(activityTooltip);

    openButton = new QPushButton("+");
    openButton->setSizePolicy(QSizePolicy::Ignored,
                              QSizePolicy::Maximum);
    openButton->setToolTip(tr("Open referenced post"));

    originalObjectMap = activity->object()->getOriginalObject();

    inReplyToMap = activity->object()->getInReplyTo();
    QString inReplyToAuthor = inReplyToMap.value("author").toMap().value("id").toString();
    if (inReplyToAuthor.startsWith("acct:"))
    {
        inReplyToAuthor.remove(0,5);
    }


    // Highlight this item if it's about the user
    bool highlightItem = false;

    // TMP/FIXME: add different kinds of highlighting
    if (activity->object()->getId() == pController->currentUserId())
    {
        highlightItem = true;
    }
    if (activity->object()->getAuthorId() == pController->currentUserId())
    {
        highlightItem = true;
    }
    if (inReplyToAuthor == pController->currentUserId())
    {
        highlightItem = true;
    }
    if (activity->getRecipientsIdList().contains("acct:" + pController->currentUserId()))
    {
        highlightItem = true;
    }

    if (highlightItem)
    {
        // Unless they are both empty!
        if (!pController->currentUserId().isEmpty())
        {
            this->setFrameStyle(QFrame::Panel);
            this->timestampLabel->setText("** "
                                          + timestampLabel->text()
                                          + " **"); // kinda TMP
        }
    }



    // Layout
    leftLayout = new QVBoxLayout();
    leftLayout->setAlignment(Qt::AlignTop);
    leftLayout->addWidget(avatarLabel, 1, Qt::AlignTop | Qt::AlignLeft);

    // Original post available (inReplyTo) or object available (note, image...)
    if (!inReplyToMap.isEmpty()
      || ((activity->object()->getType() == "note" || activity->object()->getType() == "image")
          && activity->object()->getDeletedTime().isEmpty() ) )
    {
        connect(openButton, SIGNAL(clicked()),
                this, SLOT(openOriginalPost()));
        leftLayout->addWidget(openButton, 0, Qt::AlignHCenter);
    }

    rightLayout = new QVBoxLayout();
    rightLayout->setAlignment(Qt::AlignTop);
    rightLayout->addWidget(timestampLabel);
    rightLayout->addWidget(activityDescriptionLabel);

    mainLayout = new QHBoxLayout();
    if (authorId == pController->currentUserId())
    {
        mainLayout->addLayout(rightLayout, 10);
        mainLayout->addLayout(leftLayout, 1);
    }
    else // Normal item, not ours
    {
        mainLayout->addLayout(leftLayout, 1);
        mainLayout->addLayout(rightLayout, 10);
    }
    this->setLayout(mainLayout);

    qDebug() << "MinorFeedItem created";
}



MinorFeedItem::~MinorFeedItem()
{
    qDebug() << "MinorFeedItem destroyed";
}


/*
 * Pseudo-highlight for new items
 *
 */
void MinorFeedItem::setItemAsNew(bool isNew)
{
    itemIsNew = isNew;

    if (itemIsNew)
    {
        this->setAutoFillBackground(true);
        this->setBackgroundRole(QPalette::Mid);
    }
    else
    {
        this->setAutoFillBackground(false);
        this->setBackgroundRole(QPalette::Window);

        emit itemRead();
    }
}



/*
 * On mouse click in any part of the item, set it as read
 *
 */
void MinorFeedItem::mousePressEvent(QMouseEvent *event)
{
    if (itemIsNew)
    {
        this->setItemAsNew(false);
    }

    event->accept();
}



/****************************************************************************/
/***************************** SLOTS ****************************************/
/****************************************************************************/


void MinorFeedItem::openOriginalPost()
{
    // Create a fake activity for the object
    QVariantMap originalPostMap;
    if (!inReplyToMap.isEmpty())
    {
        originalPostMap.insert("object", this->inReplyToMap);
        originalPostMap.insert("actor", this->inReplyToMap.value("author").toMap());
    }
    else
    {
        originalPostMap.insert("object", this->originalObjectMap);
        originalPostMap.insert("actor", this->originalObjectMap.value("author").toMap());
    }

    ASActivity *originalPostActivity = new ASActivity(originalPostMap, this);

    Post *referencedPost = new Post(this->pController,
                                    originalPostActivity,
                                    true, // Post is standalone
                                    this->parentWidget()); // Pass parent widget (MinorFeed) instead
                                                           // of 'this', so it won't be killed by reloads
    referencedPost->setWindowFlags(Qt::Window);
    referencedPost->setWindowModality(Qt::WindowModal);
    referencedPost->setMinimumSize(420, 360); // FIXME
    referencedPost->resize(600, 500); // FIXME
    referencedPost->show();
    referencedPost->setPostHeight();
    connect(pController, SIGNAL(commentsReceived(QVariantList,QString)),
            referencedPost, SLOT(setAllComments(QVariantList,QString)));
    referencedPost->getAllComments();
}
