# FIAIF is an Intelligent firewall
#
# description: Automates a packet filtering firewall with iptables.
#
# Script Author:	Anders Fugmann <afu at fugmann dot net>
#
# FIAIF is an Intelligent firewall
# Copyright (C) 2002-2011 Anders Peter Fugmann
#
# This program is free software; you can redistribute it and/or
# modify it under the terms of the GNU General Public License
# as published by the Free Software Foundation; either version 2
# of the License, or (at your option) any later version.
#
# This program is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public License
# along with this program; if not, write to the Free Software
# Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.

###############################################################################
# Iptables. Log errornous lines
###############################################################################
function IPTABLES ()
{
    local ERR=0
    if [ "${TEST}" = "1" ]; then
	echo "iptables $@" >> ${TEST_FILE}
    else
        iptables $@
    fi
    ERR=$?
    if (( ERR != 0 )); then
        echo "Error: iptables $@" 1>&2
        let IPTABLES_ERRORS++
    fi
    return ${ERR}
}

###############################################################################
# Call tc
###############################################################################
function TC ()
{
    local ERR=0
    if (( TEST == 1 )); then
	echo "tc $@" >> ${TEST_FILE}
    else
        tc $@ || ERR=1
    fi

    if (( ERR == 1 )); then
        echo "Error: tc $@" 1>&2
    fi
}

###############################################################################
# debug_out
# write textual messages to file if testing.
###############################################################################
function debug_out ()
{
    if (( TEST == 1 )); then
	echo "### $@" >> ${TEST_FILE}
    fi
}

###############################################################################
# print_err
# write textual messages to strdout and file if testing.
###############################################################################
function print_err ()
{
    echo "### $@"
    if (( TEST == 1 )); then
	echo "### $@" >> ${TEST_FILE}
    fi
}



###############################################################################
# Print copyright and the version number
###############################################################################
function print_version ()
{
    local VERSION=$(<${VERSION_FILE})
    echo "FIAIF ver. $VERSION, by Anders Fugmann (C) 2002-2013"
}

###############################################################################
# get_protocol_port_ip
# param:
#   <IN|OUT> [protocol [port[,port]* [ip[/mask]] ] ]
#
# returns: RETURN
###############################################################################
function get_protocol_port_ip ()
{
    local PROTOCOL=$1
    shift
    local PORTS=""
    if [[ ${PROTOCOL} == @(TCP|tcp|UDP|udp|ICMP|icmp) ]]; then
	PORTS=$1
	shift
    fi
    local IP=$1
    shift

    RETURN=""
    if [[ ${PROTOCOL} != @(ALL|all) ]]; then
	RETURN="-p ${PROTOCOL}"
	if [[ -n ${PORTS} ]]; then
	    case ${PROTOCOL} in
		TCP | tcp | UDP | udp)
	        # Test to see if the port is a range
		    if [[ ! ${PORTS} == @(ALL|all) ]]; then
			if [[ ${PORTS} == *,* ]]; then
			    RETURN="${RETURN} -m multiport --dports ${PORTS}"
			else
			    RETURN="${RETURN} --dport ${PORTS}"
			fi
		    fi
		    ;;
		ICMP|icmp)
		    if [[ ! ${PORTS} == @(ALL|all) ]]; then
			RETURN="${RETURN} --icmp-type ${PORTS}"
		    fi
		    ;;
	    esac
	fi
    fi

    local SRC_IP=${IP%*=>*}
    local DST_IP=${IP#*=>*}
    if [[ "${SRC_IP}" != */0 ]]; then
	RETURN="${RETURN} -s ${SRC_IP} "
    fi
    if [[ "${DST_IP}" != */0 ]]; then
	RETURN="${RETURN} -d ${DST_IP}"
    fi
}

###############################################################################
# Adds iptables rules, based on protocol_ip_ports
# Args: <table> <chain>
#       <protocol [port[:port]][,port[:port]]*> <ip[/mask]>=><ip[/mask]>
#       <rule>
###############################################################################
function add_rule_protocol_ip_port
{
    local TABLE=$1
    local CHAIN=$2
    local PROTOCOL=$3
    shift 3
    local PORTS=""
    if [[ ${PROTOCOL} == @(TCP|tcp|UDP|udp|ICMP|icmp) ]]; then
	PORTS=$1
	shift
    fi
    local IP=$1
    shift
    local RULE=$@

    local SRC_IP=${IP/=>*}
    local DST_IP=${IP//*=>}
    if [[ ${SRC_IP} == *([[:alnum:]_]) ]]; then
	# Expand SRC_IP
	local VAR=IPSET_${SRC_IP}
	if [[ -f "${CONF_DIR}/${!VAR}" ]]; then
	    cat ${CONF_DIR}/${!VAR} | cut -d"#" -f1 | \
		while read SRC_IP; do
		if [[ -z ${SRC_IP} ]]; then
		    continue
		fi
		add_rule_protocol_ip_port ${TABLE} ${CHAIN} \
		    ${PROTOCOL} ${PORTS} ${SRC_IP}=\>${DST_IP} ${RULE}
	    done
	else
	    for SRC_IP in ${!VAR}; do
		add_rule_protocol_ip_port ${TABLE} ${CHAIN} \
		    ${PROTOCOL} ${PORTS} ${SRC_IP}=\>${DST_IP} ${RULE}
	    done
	fi
	# End processing
	return

    elif [[ ${DST_IP} == *([[:alnum:]_]) ]]; then
	# Expand DST_IP
	local VAR=IPSET_${DST_IP}
	if [[ -f "${CONF_DIR}/${!VAR}" ]]; then
	    cat ${CONF_DIR}/${!VAR} | cut -d"#" -f1 | \
		while read DST_IP; do
		if [[ -z ${DST_IP} ]]; then
		    continue
		fi
		add_rule_protocol_ip_port ${TABLE} ${CHAIN} \
		    ${PROTOCOL} ${PORTS} ${SRC_IP}=\>${DST_IP} ${RULE}
	    done
	else
	    for DST_IP in ${!VAR}; do
		add_rule_protocol_ip_port ${TABLE} ${CHAIN} \
		    ${PROTOCOL} ${PORTS} ${SRC_IP}=\>${DST_IP} ${RULE}
	    done
	fi
	# End processing
	return
    fi

    # Split the ports if nessesary

    if [[ ${PROTOCOL} == @(TCP|tcp|UDP|udp) && -n "${PORTS}" &&
	   ${PORTS} != +([[:alnum:]]):+([[:alnum:]]) &&
	   ${PORTS} != +([[:alnum:]-])*(,+([[:alnum:]-])) ]]; then
	local SINGLE_PORTS=""
	local RANGE_PORTS=""
	for ELEM in ${PORTS//,/ }; do
	    if [[ ${ELEM} == +([[:alnum:]]):+([[:alnum:]]) ]]; then
		RANGE_PORTS="${RANGE_PORTS} ${ELEM}"
	    else
		SINGLE_PORTS="${SINGLE_PORTS},${ELEM}"
	    fi
	done


	for PORTS in ${RANGE_PORTS} ${SINGLE_PORTS#,}; do
	    add_rule_protocol_ip_port ${TABLE} ${CHAIN} \
		${PROTOCOL} ${PORTS} ${IP} ${RULE}
	done
	return

    fi

    # Do not allow more than MAX_MULTI_PORTS on multiport argument
    declare -a PORT_ARR=( ${PORTS//,/ } )
    if (( ${#PORT_ARR[*]} > MAX_MULTI_PORTS )); then
	local I
	PORTS=""
	for ((I=0;I<${#PORT_ARR[*]};I++)); do
	    if (( I % MAX_MULTI_PORTS == 0 )) && [[ -n ${PORTS} ]]; then
		add_rule_protocol_ip_port ${TABLE} ${CHAIN} \
		    ${PROTOCOL} ${PORTS#,} ${IP} ${RULE}
		PORTS=""
	    fi
	    PORTS="${PORTS},${PORT_ARR[I]}"
	done
	PORTS=${PORTS#,}
	if [[ -z "${PORTS}" ]]; then
	    return
	fi
    fi

    if [[ -n "${PORTS}" && ${PROTOCOL} == @(ICMP|icmp) ]]; then
	declare -a ICMP_TYPES=( ${PORTS//,/ } )
	local I
	for ((I=0;I<${#ICMP_TYPES[*]}-1;I++)); do
	    add_rule_protocol_ip_port ${TABLE} ${CHAIN} \
		${PROTOCOL} ${ICMP_TYPES[I]} ${IP} ${RULE}
	done
	PORTS=${ICMP_TYPES[I]}
    fi

    get_protocol_port_ip ${PROTOCOL} ${PORTS} ${IP}
    local PROT_IP_PORT=${RETURN}

    # Call iptables
    IPTABLES -t ${TABLE} -A ${CHAIN} ${PROT_IP_PORT} ${RULE}

}

###############################################################################
# Give staus of the firewall
###############################################################################
function iptables_status ()
{
	IPTABLES -nvL
	IPTABLES -t nat -nvL
	IPTABLES -t mangle -nvL
}

###############################################################################
# Restore the firewall rules as saved with save_rules
###############################################################################
function restore_rules ()
{
    local STATE_FILE=$1
    echo -n "Restoring rules: "
    if [[ -f ${STATE_FILE} ]]; then
	iptables-restore -c < ${STATE_FILE}
    else
	echo -n "File '${STATE_FILE}' not found "
    fi
    echo "Done."
}

###############################################################################
# Give staus of the firewall
###############################################################################
function save_rules ()
{
    local STATE_FILE=$1
    echo -n "Saving rules: "
    # Make sure that this file is not tampered.
    rm -f ${STATE_FILE}
    touch ${STATE_FILE}
    chown root:root ${STATE_FILE}
    chmod 600 ${STATE_FILE}
    iptables-save -c >> ${STATE_FILE}
    echo "Done."

}

###############################################################################
# apply_script:
###############################################################################
function apply_script
{
    local SCRIPT=$1
    local LENGTH=$2
    local VAR
    local I
    for ((I=0;I<${LENGTH};I++)); do
	VAR=${SCRIPT}[$I]
	debug_out "${VAR}=${!VAR}"
        if (( TEST == 1 )); then
	    echo "${!VAR}" >> ${TEST_FILE}
	else
	    ${!VAR}
	fi
    done
}

###############################################################################
# Get the name of a service based on port and protocol
# RESULT holds the return value.
# The result is cached.
###############################################################################
function get_service_name
{
    local PROTOCOL=$1
    local PORT=$2
    #Lookup in the cache
    RESULT=""
    local I
    if [[ -n "${PORT}" ]]; then
	for ((I=0;I<${#SERVICE_CACHE[*]};I++));do
	    if [[ "${SERVICE_CACHE[I]%:*}" = "${PORT}/${PROTOCOL}" ]];then
		RESULT=${SERVICE_CACHE[I]#*:}
		return
	    fi
	done

	RESULT=$(grep -i -e "[^0-9]${PORT}/${PROTOCOL}" ${SERVICES} | cut -f1)
	if [[ -z "${RESULT}" ]]; then
	    RESULT="${PORT}"
	fi
	SERVICE_CACHE[${#SERVICE_CACHE[*]}]="${PORT}/${PROTOCOL}:$RESULT"
    else
	RESULT=""
    fi
}

###############################################################################
# Get the port number based on a service name and protocol
# RESULT holds the return value.
###############################################################################
function get_service_port ()
{
    local PROTOCOL=$1
    local PORTNAME=$2
    RESULT=$(grep -e "${PORTNAME}.*/${PROTOCOL}" /etc/services | cut -f 3 | cut -d'/' -f1)
}
###############################################################################
# Get the name of a ip-address.
# RESULT holds the return value.
# Result is pooled for later use.
###############################################################################
function get_host_name ()
{
    local IP=$1
    RESULT=${IP}
    if [[ -n "${IP}" ]]; then
	local I
	for ((I=0;I<${#HOST_CACHE[*]};I++));do
	    if [[ "${HOST_CACHE[I]}" = ${IP}:* ]];then
		RESULT=${HOST_CACHE[I]#*:}
		return
	    fi
	done
	RESULT=$(${DNS_RESOLVE} ${IP})
	if [[ -n "${RESULT}" ]]; then
	    IP=${RESULT%%.}
	fi
	# Always put result into the cache to save time looking up again.
	HOST_CACHE[${#HOST_CACHE[*]}]="${IP}:$RESULT"
    fi
}

###############################################################################
# Load modules.
###############################################################################
function load_modules ()
{
    local MODULE
    for MODULE in ${MODULES}; do
	if (( TEST == 1 )); then
	    echo "modprobe ${MODULE}" >> ${TEST_FILE}
	else
	    modprobe ${MODULE}
	fi
    done
}

###############################################################################
# Unload modules.
###############################################################################
function unload_modules ()
{
    local MODULE
    for MODULE in ${MODULES}; do
	if (( TEST == 1 )); then
	    echo "modprobe -r ${MODULE}" >> ${TEST_FILE}
	else
	    modprobe -r ${MODULE} > /dev/null 2>&1
	fi
    done
}

###############################################################################
# Convert a mask to a number.
###############################################################################
function mask_to_number ()
{
    local MASK=$1
    RESULT=1
    case ${MASK} in
	1)
	    RESULT=128
	    ;;
	2)
	    RESULT=64
	    ;;
	3)
	    RESULT=32
	    ;;
	4)
	    RESULT=16
	    ;;
	5)
	    RESULT=8
	    ;;
	6)
	    RESULT=4
	    ;;
	7)
	    RESULT=2
	    ;;
	8)
	    RESULT=1
	    ;;
    esac
}
###############################################################################
# Convert a mask to a number.
###############################################################################
function bitmap_to_mask ()
{
    local MASK=$1
    RESULT=0
    local BITS
    local i
    for ((i=0;i<4;i++)); do

	case ${MASK%%.*} in
	    255)
		BITS=8
		;;
	    254)
		BITS=7
		;;
	    252)
		BITS=6
		;;
	    248)
		BITS=5
		;;
	    240)
		BITS=4
		;;
	    224)
		BITS=3
		;;
	    192)
		BITS=2
		;;
	    128)
		BITS=1
		;;
	    0)
		BITS=0
	    ;;
	esac

	RESULT=$(( ${RESULT} + ${BITS} ))
	MASK=${MASK#*.}
    done
}

###############################################################################
# Test if an IP is in the given network
# Param: network mask ip
# Where mask is a single number.
###############################################################################
function ip_in_network ()
{
    local NET=$1
    local MASK=$2
    local IP=$3

    local NUMBER
    # If mask is in dotted notation, convert it to bitmask
    if [[ "${MASK}" == +(+([[:digit:]]).)+([[:digit:]]) ]]; then
	bitmap_to_mask ${MASK}
	MASK=${RESULT}
    fi

    while (( ${MASK} >= 1 )); do
	mask_to_number ${MASK}
	NUMBER=${RESULT}

	NET_FIELD=$(( ${NET%%.*} / ${NUMBER} ))
	IP_FIELD=$(( ${IP%%.*} / ${NUMBER} ))

	#echo ${NET_FIELD} -ne ${IP_FIELD}
	if (( NET_FIELD != IP_FIELD )); then
	    return -1
	    break
	fi

	NET=${NET#*.}
	IP=${IP#*.}
	MASK=$((${MASK} - 8))
    done
    return 0
}

###############################################################################
# Compare two ip numbers.
# Result: -1 ip1 lower than ip2
#          0 equal
#          1 ip1 greater than ip2
###############################################################################
function compare_ip ()
{
    RESULT=0

    declare -a IP1="( $(echo $1 | sed s/\\\./\ /g) )"
    declare -a IP2="( $(echo $2 | sed s/\\\./\ /g) )"
    local I
    for ((I=0;I<4;I++)); do
	if (( ${IP1[I]} > ${IP2[I]} ));then
	    RESULT=1
	    return
	elif (( ${IP1[I]} < ${IP2[I]} )); then
	    RESULT=-1
	    return
	fi
    done
}
###############################################################################
# Check that argument is an ip number
# return value 0 if argument is a number
###############################################################################
function check_ip ()
{
    local IP=$1
    if [[ "${IP}" == +([[:digit:]]).+([[:digit:]]).+([[:digit:]]).+([[:digit:]]) ]]; then
	return 0
    else
	return 1
    fi
}

###############################################################################
# Check that a zone exists
# function returns 0 on success
###############################################################################
function check_zone ()
{
    local ZONE=$1
    local ERRORMSG=$2
    RESULT=0
    local AZONE

    for AZONE in ${ZONES} ALL; do
	if [[ "${ZONE}" == "${AZONE}" ]]; then
	    return 0
	fi
    done
    if [[ -n "${ERRORMSG}" ]]; then
	local ZONEFILE
	ZONEFILE=CONF_${NAME}
	ZONEFILE=${!ZONEFILE}
	echo -e "\n${ZONEFILE}: ${ERRORMSG}\n"
	debug_out "${ZONEFILE}: ${ERRORMSG}"
    fi
    return 1
}

###############################################################################
# Locate zone based on ip and interface. All zones needs to be loaded.
# Param: <interface> <ip>
# Returns: Name of the zone
###############################################################################
function get_zone_name ()
{
    local ZONE_DEV=$1
    local ZONE_IP=$2

    RESULT="UNKNOWN"

    for ZONE in ${ZONES}; do
	GLOBAL=${ZONE}_GLOBAL; GLOBAL=${!GLOBAL}
	DYNAMIC=${ZONE}_DYNAMIC; DYNAMIC=${!DYNAMIC}
	NETS=${ZONE}_NETS; NETS=${!NETS}

	if [[ ${ZONE_DEV} == ${DEV} ]]; then
	    if (( GLOBAL == 1 || DYNAMIC == 1 )); then
		RESULT=${ZONE}
	    else
		for NET in ${NETS}; do
		    if ip_in_network ${NET%/*} ${NET#*/} ${ZONE_IP}; then
			RESULT=${ZONE}
		    fi
		done
	    fi
	fi
	if [[ "${RESULT}" != "UNKNOWN" ]]; then
	    break
	fi
    done
}

###############################################################################
# Jumps to the specified chain, if the destination zone matches.
# Args: <zone> <SRC|DST> <table> <old_chain> <new_chain>
###############################################################################
function jump_zone ()
{
    local DST_ZONE=$1
    local DIRECTION=$2
    local TABLE=$3
    local SRC_CHAIN=$4
    local DST_CHAIN=$5

    local NETS=${DST_ZONE}_NETS
    local GLOBAL=${DST_ZONE}_GLOBAL
    local DYNAMIC=${DST_ZONE}_DYNAMIC
    local DEV=${DST_ZONE}_DEV
    NETS=${!NETS}
    GLOBAL=${!GLOBAL}
    DYNAMIC=${!DYNAMIC}
    DEV=${!DEV}

    local DEV_ARG
    local NET_ARG
    case ${DIRECTION} in
	SRC)
	    DEV_ARG="-i"
	    NET_ARG="-s"
	    ;;
	DST)
	    DEV_ARG="-o"
	    NET_ARG="-d"
	    ;;
    esac

    if (( GLOBAL == 1 || DYNAMIC == 1 )); then
	IPTABLES -t ${TABLE} -A ${SRC_CHAIN} ${DEV_ARG} ${DEV} -j ${DST_CHAIN}
    else
	local NET
	for NET in ${NETS}; do
	    IPTABLES -t ${TABLE} -A ${SRC_CHAIN} \
		${DEV_ARG} ${DEV} ${NET_ARG} ${NET} -j ${DST_CHAIN}
	done
    fi
}

###############################################################################
# Test if the current kernel version is greater or equal to the given.
# E.g: kernel_version 2.4.21
###############################################################################
function kernel_version_ge ()
{
    local KERNEL_VERSION=$1
    local KMA=$(echo ${KERNEL_VERSION} | cut -d"." -f 1)
    local KMI=$(echo ${KERNEL_VERSION} | cut -d"." -f 2)
    local KRE=$(echo ${KERNEL_VERSION} | cut -d"." -f 3)

    local VAL=$(( ${KMA}*10000 + ${KMI}*100 + ${KRE} ))
    local CUR_VAL=$(( ${MAJOR}*10000 + ${MINOR}*100 + ${RELEASE}))

    if (( ${CUR_VAL} >= ${VAL} )); then
	return 0
    else
	return 1
    fi
}

###############################################################################
# Test if the current kernel version is less or equal to the given.
# E.g: kernel_version 2.4.21
###############################################################################
function kernel_version_le ()
{
    local KERNEL_VERSION=$1
    local KMA=$(echo ${KERNEL_VERSION} | cut -d"." -f 1)
    local KMI=$(echo ${KERNEL_VERSION} | cut -d"." -f 2)
    local KRE=$(echo ${KERNEL_VERSION} | cut -d"." -f 3)

    local VAL=$(( ${KMA}*10000 + ${KMI}*100 + ${KRE} ))
    local CUR_VAL=$(( ${MAJOR}*10000 + ${MINOR}*100 + ${RELEASE} ))
    if (( ${CUR_VAL} <= ${VAL} )); then
	return 0
    else
	return 1
    fi
}
