/*
 * @(#)ReadByteStream.java    1.0.0 04/13/2001 - 11:44:03
 *
 * Copyright (C) 2001,,2003 2002 Matt Albrecht
 * groboclown@users.sourceforge.net
 * http://groboutils.sourceforge.net
 *
 *  Permission is hereby granted, free of charge, to any person obtaining a
 *  copy of this software and associated documentation files (the "Software"),
 *  to deal in the Software without restriction, including without limitation
 *  the rights to use, copy, modify, merge, publish, distribute, sublicense,
 *  and/or sell copies of the Software, and to permit persons to whom the
 *  Software is furnished to do so, subject to the following conditions:
 *
 *  The above copyright notice and this permission notice shall be included in
 *  all copies or substantial portions of the Software.
 *
 *  THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
 *  IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
 *  FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT.  IN NO EVENT SHALL
 *  THE AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
 *  LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING
 *  FROM, OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER
 *  DEALINGS IN THE SOFTWARE.
 */
 

package net.sourceforge.groboutils.util.io.v1;

import java.io.File;
import java.io.IOException;
import java.io.ByteArrayOutputStream;
import java.io.InputStream;





/**
 * Reads a byte array from a stream until the stream is finished.
 * You can specify a maximum size to read, and the block read size.
 *
 * @author   Matt Albrecht <a href="mailto:groboclown@users.sourceforge.net">groboclown@users.sourceforge.net</a>
 * @since    April 13, 2001
 * @version  $Date: 2003/05/19 20:31:47 $
 */
public class ReadByteStream
{
    //----------------------------
    // Public data
    
    /**
     * Read in an unlimited number of bytes.  This can be very
     * dangerous.
     */
    public static final int READ_TO_END_OF_STREAM = Integer.MAX_VALUE;
    
    /**
     * Default block read size.
     */
    public static final int DEFAULT_BLOCK_READ_SIZE = 4096;
    
    //----------------------------
    // Private data
    
    private InputStream m_is;
    private int m_maxSize;
    private int m_bufferSize;
    
    
    //----------------------------
    // constructors
    
    /**
     * Default constructor
     */
    public ReadByteStream( InputStream is )
    {
        this( is, READ_TO_END_OF_STREAM, DEFAULT_BLOCK_READ_SIZE );
    }
    
    
    /**
     * 
     */
    public ReadByteStream( InputStream is, int maxReadSize, int blockReadSize )
    {
        setInputStream( is );
        setSizes( maxReadSize, blockReadSize );
    }
    
    
    //----------------------------
    // Public methods
    
    
    /**
     * Sets the internal input stream.
     */
    public void setInputStream( InputStream is )
    {
        if (is == null)
        {
            throw new IllegalArgumentException( "input stream is null" );
        }
        this.m_is = is;
    }
    
    
    /**
     * Sets the internal sizes.
     */
    public void setSizes( int maxReadSize, int blockReadSize )
    {
        if (blockReadSize <= 0)
        {
            blockReadSize = DEFAULT_BLOCK_READ_SIZE;
        }
        if (maxReadSize <= 0 || maxReadSize > READ_TO_END_OF_STREAM)
        {
            maxReadSize = READ_TO_END_OF_STREAM;
        }
        if (maxReadSize < blockReadSize)
        {
            blockReadSize = maxReadSize;
        }
        this.m_maxSize = maxReadSize;
        this.m_bufferSize = blockReadSize;
    }
    
    
    /**
     * Read in the byte stream, using the current settings.
     */
    public byte[] readByteStream()
            throws IOException
    {
        return readByteStream( this.m_is, this.m_maxSize, this.m_bufferSize );
    }
    
    
    /**
     * Read in the byte stream.  Does not close the stream after it has
     * finished reading.  Uses the default sizes.
     *
     * @see #readByteStream( InputStream, int, int )
     */
    public static byte[] readByteStream( InputStream is )
            throws IOException
    {
        return readByteStream( is, READ_TO_END_OF_STREAM,
            DEFAULT_BLOCK_READ_SIZE );
    }
    
    
    /**
     * Read in the byte stream.  Does not close the stream after it has
     * finished reading.
     * <P>
     * Note that there is no variable checking, for performance reasons.
     * The user needs to verify that:
     * @param is the input stream, which cannot be <tt>null</tt>.
     * @param maxReadSize the maximum number of bytes to read, which
     *      must be positive, and must be modulo 0 of <tt>blockReadSize</tt>.
     *      This is an "estimation", and may actually read in more than this
     *      many bytes if it is not modulo 0 of <tt>blockReadSize</tt>, but
     *      will always return all the bytes read.
     * @param blockReadSize the number of bytes to read in per read command,
     *      which cannot be more than <tt>maxReadSize</tt>, and cannot be
     *      less than or equal to zero.
     */
    public static byte[] readByteStream( InputStream is, int maxReadSize,
            int blockReadSize )
            throws IOException
    {
        ByteArrayOutputStream baos = new ByteArrayOutputStream();
        byte buffer[] = new byte[ blockReadSize ];
        int size = is.read( buffer, 0, blockReadSize );
        int totSize = size;
        while (size > 0 && totSize < maxReadSize)
        {
            baos.write( buffer, 0, size );
            size = is.read( buffer, 0, blockReadSize );
            totSize += size;
        }
        baos.close();
        return baos.toByteArray();
    }
}
 
