// Flags: --security-revert=CVE-2021-44531
'use strict';
const common = require('../common');

if (!common.hasCrypto)
  common.skip('missing crypto');

const assert = require('assert');
const util = require('util');

const tls = require('tls');

common.expectWarning('DeprecationWarning', [
  ['The URI http://[a.b.a.com]/ found in cert.subjectaltname ' +
  'is not a valid URI, and is supported in the tls module ' +
  'solely for compatibility.',
   'DEP0109'],
]);

const tests = [
  // Likewise for "URI:" Subject Alternative Names.
  // See also https://github.com/nodejs/node/issues/8108.
  {
    host: '8.8.8.8',
    cert: { subject: { CN: '8.8.8.8' }, subjectaltname: 'URI:http://8.8.8.8/' },
    error: 'IP: 8.8.8.8 is not in the cert\'s list: '
  },
  // Empty Subject w/URI name
  {
    host: 'a.b.a.com', cert: {
      subjectaltname: 'URI:http://a.b.a.com/',
    }
  },
  // URI names
  {
    host: 'a.b.a.com', cert: {
      subjectaltname: 'URI:http://a.b.a.com/',
      subject: {}
    }
  },
  {
    host: 'a.b.a.com', cert: {
      subjectaltname: 'URI:http://*.b.a.com/',
      subject: {}
    },
    error: 'Host: a.b.a.com. is not in the cert\'s altnames: ' +
           'URI:http://*.b.a.com/'
  },
  // Invalid URI
  {
    host: 'a.b.a.com', cert: {
      subjectaltname: 'URI:http://[a.b.a.com]/',
      subject: {}
    }
  },
];

tests.forEach(function(test, i) {
  const err = tls.checkServerIdentity(test.host, test.cert);
  assert.strictEqual(err && err.reason,
                     test.error,
                     `Test# ${i} failed: ${util.inspect(test)} \n` +
                     `${test.error} != ${(err && err.reason)}`);
});
