(* This file is part of our reusable OCaml BRICKS library
   Copyright (C) 2010  Jean-Vincent Loddo

   This program is free software: you can redistribute it and/or modify
   it under the terms of the GNU General Public License as published by
   the Free Software Foundation, either version 2 of the License, or
   (at your option) any later version.

   This program is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
   GNU General Public License for more details.

   You should have received a copy of the GNU General Public License
   along with this program.  If not, see <http://www.gnu.org/licenses/>. *)

(** Generate a file <source>.ml.pot during the <source>.ml parsing. *)

open Camlp4 (* -*- camlp4o -*- *)

(* ocamlc -I +camlp4 -pp camlp4of.opt camlp4lib.cma gettext_extract_pot_p4.ml *)

module Id = struct
  let name = "gettext_extract_pot_p4.ml"
  let version = "$Id: gettext_extract_pot_p4,v 0.1 2009/04/10 16:16:16 $"
end

let pp_name = "gettext_extract_pot_p4"

module Tool = struct
 INCLUDE "CAMLP4/common_tools_for_preprocessors.ml"
end

let header =
 let file = "gettext_extract_pot_p4.conf" in
 let project_id_version = Tool.Conf.conf file ~default:"project_id_version???" "project_id_version" in
 let report_bugs_to     = Tool.Conf.conf file ~default:"report_bugs_to???" "report_bugs_to" in
 let charset            = Tool.Conf.conf file ~default:"utf-8" "charset" in
 Printf.sprintf
"# Copyright (C) OWNER
# AUTHOR, YEAR.
#
msgid \"\"
msgstr \"\"
\"Project-Id-Version: %s\\n\"
\"Report-Msgid-Bugs-To: %s\\n\"
\"Content-Type: text/plain; charset=%s\\n\"
" project_id_version report_bugs_to charset


module Make (Syntax : Sig.Camlp4Syntax) = struct
  open Sig
  include Syntax

  let append =
   let fd_ref = ref None in       (* In order to open the file once *)
   let ht = Hashtbl.create 251 in (* In order to avoid duplicates *)
   fun _loc ?(msgstr="") msgid ->
     if Hashtbl.mem ht msgid then () else
     begin
      Hashtbl.add ht msgid ();
      let fd = match !fd_ref with
      | None    ->
         let fname = ((Loc.file_name _loc)^".pot") in
         let fd = open_out fname in
         (fd_ref := Some fd);
         (output_string fd header);
         fd
      | Some fd -> fd
      in
      let line  = Loc.start_line _loc in
      let fname = Loc.file_name  _loc in
      let content =
Printf.sprintf "
#. generated by %s
#: %s:%d
msgid \"%s\"
msgstr \"%s\"\n" pp_name fname line msgid msgstr
     in
      (output_string fd content);
      (flush fd)
     end

  EXTEND Gram
    GLOBAL: expr;

    gettext_id : [[ LIDENT "s_" -> "s_" ] | [ LIDENT "f_" -> "f_" ]];

    expr: FIRST
      [[ id = gettext_id; msgid = STRING ; msgstr = STRING ->
         let () = append _loc msgid ~msgstr in
         <:expr< $lid:id$ $str:msgid$ >>

       | id = gettext_id; msgid = STRING ->
         let () = append _loc msgid in
         <:expr< $lid:id$ $str:msgid$ >>
         ]];
  END

end

let module M = Register.OCamlSyntaxExtension (Id) (Make) in ()
