///////////////////////////////////////////////////////////////////////////////
// 
//  Copyright (2008) Alexander Stukowski
//
//  This file is part of OVITO (Open Visualization Tool).
//
//  OVITO is free software; you can redistribute it and/or modify
//  it under the terms of the GNU General Public License as published by
//  the Free Software Foundation; either version 2 of the License, or
//  (at your option) any later version.
//
//  OVITO is distributed in the hope that it will be useful,
//  but WITHOUT ANY WARRANTY; without even the implied warranty of
//  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
//  GNU General Public License for more details.
//
//  You should have received a copy of the GNU General Public License
//  along with this program.  If not, see <http://www.gnu.org/licenses/>.
//
///////////////////////////////////////////////////////////////////////////////

/** 
 * \file PathManager.h 
 * \brief Contains the definition of the Core::ParameterUnit class
 *        and its derived classes. 
 */

#ifndef __OVITO_PARAMETER_UNIT_H
#define __OVITO_PARAMETER_UNIT_H

#include <core/Core.h>
#include <core/scene/animation/AnimManager.h>

namespace Core {

/**
 * \brief Base class for parameter unit conversion services. 
 * 
 * A ParameterUnit is used to convert a controller value from the controller's native units to
 * another unit presented to the user and vice versa. One example for
 * a ParameterUnit is the AngleUnit class which converts between radians and degrees.
 * 
 * \author Alexander Stukowski
 * \sa UnitsManager
 */
class CORE_DLLEXPORT ParameterUnit : public PluginClass 
{
protected:
	
	/// \brief Default constructor.
	ParameterUnit() : PluginClass() {}
    
public:

	/// \brief Converts a value from native units to the units presented to the user.
	/// \param nativeValue The value in internal units to be converted.
	/// \return The value as converted to the user units.
	/// \sa userToNative()
	virtual FloatType nativeToUser(FloatType nativeValue) = 0;
	
	/// \brief Converts a value from user units to the native units used internally.
	/// \param userValue The value to be converted back to internal units.
	/// \return The converted value.
	/// \sa nativeToUser()
	virtual FloatType userToNative(FloatType userValue) = 0;
	
	/// \brief Converts the given string to a value.
	/// \param valueString This is a string representation of a value as it might have
	///                    been produced by formatValue() or entered by the user.
	/// \return The parsed value in user units.
	/// \throw Exception when the value could not be parsed.
	/// \sa formatValue()
	virtual FloatType parseString(const QString& valueString) = 0;
	
	/// \brief Converts a numeric value to a string.
	/// \param value The value to be converted. This is in user units.
	/// \return The string representation of the value. This can be converted back using parseString().
	/// \sa parseString()
	virtual QString formatValue(FloatType value) = 0;
	
	/// \brief Returns the step size used by spinner widgets for this paremeter unit type.
	/// \param currentValue The current value of the spinner in native units. This can be used to make the step size value dependent.
	/// \param upDirection Specifies whether the spinner is dragged in the positive or the negative direction.
	/// \return The numeric step size used by SpinnerWidget for this parameter type. This is in native units.
	/// 
	/// The default implementation just returns 1.
	virtual FloatType stepSize(FloatType currentValue, bool upDirection) { return 1; }

Q_SIGNALS:

	/// \brief This signal is emitted by the parameter unit when the display
	///        format or conversion factor has changed and the value to string
	///        conversion has to be re-done.
	void formatChanged();
	
private:
	
	Q_OBJECT
	DECLARE_ABSTRACT_PLUGIN_CLASS(ParameterUnit)
};

/**
 * \brief Default parameter unit that is used by float controllers that
 *        have no custom ParameterUnit class assigned. 
 * 
 * It does no unit conversion at all. Values are formatted as floating-point strings.
 * 
 * \author Alexander Stukowski
 */
class CORE_DLLEXPORT FloatParameterUnit : public ParameterUnit
{
public:
	
	/// \brief Default constructor.
	FloatParameterUnit() : ParameterUnit() {}    

	/// \brief Converts a value from native units to the units presented to the user.
	/// \param nativeValue The value in internal units to be converted.
	/// \return The value as converted to the user units.
	///
	/// This implementation just returns the unmodified input value.
	/// \sa userToNative()
	virtual FloatType nativeToUser(FloatType nativeValue) { return nativeValue; }
	
	/// \brief Converts a value from user units to the native units used internally.
	/// \param userValue The value to be converted back to internal units.
	/// \return The converted value.
	///
	/// This default implementation just returns the unmodified input value.
	/// \sa nativeToUser()
	virtual FloatType userToNative(FloatType userValue) { return userValue; }

	/// \brief Converts the given string to a value.
	/// \param valueString This is a string representation of a value as it might have
	///                    been produced by formatValue() or entered by the user.
	/// \return The parsed value in user units.
	/// \throw Exception when the value could not be parsed.
	/// \sa formatValue()
	virtual FloatType parseString(const QString& valueString) {
		double value;
		bool ok;
		value = valueString.toDouble(&ok);		
		if(!ok)
			throw Exception(tr("Invalid floating point value: %1").arg(valueString));
		return (FloatType)value;
	}
		
	/// \brief Converts a numeric value to a string.
	/// \param value The value to be converted. This is in user units.
	/// \return The string representation of the value. This can be converted back using parseString().
	/// \sa parseString()
	virtual QString formatValue(FloatType value) {
		return QString::number(value);
	}

	/// \brief Returns the step size used by spinner widgets for this paremeter unit type.
	/// \param currentValue The current value of the spinner in native units. This can be used to make the step size value dependent.
	/// \param upDirection Specifies whether the spinner is dragged in the positive or the negative direction.
	/// \return The numeric step size used by SpinnerWidget for this parameter type. This is in native units.
	virtual FloatType stepSize(FloatType currentValue, bool upDirection) {
		int exponent;
		currentValue = nativeToUser(currentValue);
		if(currentValue != 0) {
			if(upDirection ^ (currentValue < 0.0))
				exponent = (int)floor(log10(abs(currentValue))-1.0);
			else
				exponent = (int)floor(log10(abs(currentValue))-1.0001);
			if(exponent < -5) exponent = -5;
			else if(exponent > 5) exponent = 5;
		}
		else exponent = 0;
		return userToNative(pow(10.0, exponent));
	}
 
private:
	Q_OBJECT
	DECLARE_PLUGIN_CLASS(FloatParameterUnit)
};

/**
 * \brief Default parameter unit that is used by integer controllers fields that
 *        have no custom ParameterUnit class assigned. 
 * 
 * It does no unit conversion at all. Values are formatted as integer value strings.
 * 
 * \author Alexander Stukowski
 */
class CORE_DLLEXPORT IntegerParameterUnit : public ParameterUnit
{
public:
	
	/// \brief Default constructor.
	IntegerParameterUnit() : ParameterUnit() {}    

	/// \brief Converts a value from native units to the units presented to the user.
	/// \param nativeValue The value in internal units to be converted.
	/// \return The value as converted to the user units.
	///
	/// This implementation just returns the unmodified input value.
	/// \sa userToNative()
	virtual FloatType nativeToUser(FloatType nativeValue) { return nativeValue; }
	
	/// \brief Converts a value from user units to the native units used internally.
	/// \param userValue The value to be converted back to internal units.
	/// \return The converted value.
	///
	/// This default implementation just returns the unmodified input value.
	/// \sa nativeToUser()
	virtual FloatType userToNative(FloatType userValue) { return userValue; }

	/// \brief Converts the given string to a value.
	/// \param valueString This is a string representation of a value as it might have
	///                    been produced by formatValue() or entered by the user.
	/// \return The parsed value in user units.
	/// \throw Exception when the value could not be parsed.
	/// \sa formatValue()
	virtual FloatType parseString(const QString& valueString) {
		int value;
		bool ok;
		value = valueString.toInt(&ok);		
		if(!ok)
			throw Exception(tr("Invalid integer value: %1").arg(valueString));
		return (FloatType)value;
	}
		
	/// \brief Converts a numeric value to a string.
	/// \param value The value to be converted. This is in user units.
	/// \return The string representation of the value. This can be converted back using parseString().
	/// \sa parseString()
	virtual QString formatValue(FloatType value) {
		return QString::number((int)value);
	}

private:
	Q_OBJECT
	DECLARE_PLUGIN_CLASS(IntegerParameterUnit)
};

/*
 * \brief This ParameterUnit is used by parameter values that specify a distance or a position in space.
 * 
 * \author Alexander Stukowski
 */
class CORE_DLLEXPORT WorldParameterUnit : public FloatParameterUnit
{
public:
	/// \brief Default constructor.
	WorldParameterUnit() : FloatParameterUnit() {}    

private:
	Q_OBJECT
	DECLARE_PLUGIN_CLASS(WorldParameterUnit)
};

/**
 * \brief This ParameterUnit implementation converts between radians and degrees.
 * 
 * \author Alexander Stukowski
 */
class CORE_DLLEXPORT AngleParameterUnit : public FloatParameterUnit
{
public:
	
	/// \brief Default constructor.
	AngleParameterUnit() : FloatParameterUnit() {}    

	/// \brief Converts a value from native units to the units presented to the user.
	/// \param nativeValue The value in internal units to be converted.
	/// \return The value as converted to the user units.
	///
	/// This implementation converts from radians to degrees.
	/// \sa userToNative()
	virtual FloatType nativeToUser(FloatType nativeValue) { return nativeValue * (180.0 / FLOATTYPE_PI); }
	
	/// \brief Converts a value from user units to the native units used internally.
	/// \param userValue The value to be converted back to internal units.
	/// \return The converted value.
	///
	/// This default implementation converts from degrees to radians.
	/// \sa nativeToUser()
	virtual FloatType userToNative(FloatType userValue) { return userValue * (FLOATTYPE_PI / 180.0); }

private:
	
	Q_OBJECT
	DECLARE_PLUGIN_CLASS(AngleParameterUnit)
};

/**
 * \brief This ParameterUnit implementation is used for percentage values.
 * 
 * \author Alexander Stukowski
 */
class CORE_DLLEXPORT PercentParameterUnit : public FloatParameterUnit
{
public:
	
	/// \brief Default constructor.
	PercentParameterUnit() : FloatParameterUnit() {}    

	/// \brief Converts a value from native units to the units presented to the user.
	/// \param nativeValue The value in internal units to be converted.
	/// \return The value as converted to the user units.
	///
	/// This implementation converts from the [0,1] range to the [0,100] percent range.
	/// \sa userToNative()
	virtual FloatType nativeToUser(FloatType nativeValue) { return nativeValue * 100.0; }
	
	/// \brief Converts a value from user units to the native units used internally.
	/// \param userValue The value to be converted back to internal units.
	/// \return The converted value.
	///
	/// This default implementation converts from the [0,100] percent range to the [0,1] range.
	/// \sa nativeToUser()
	virtual FloatType userToNative(FloatType userValue) { return userValue / 100.0; }

private:
	
	Q_OBJECT
	DECLARE_PLUGIN_CLASS(PercentParameterUnit)
};

/**
 * \brief This ParameterUnit is used by parameter values that specify a time value.
 * 
 * \author Alexander Stukowski
 */
class CORE_DLLEXPORT TimeParameterUnit : public IntegerParameterUnit
{
public:
	
	/// \brief Default constructor.
	TimeParameterUnit() : IntegerParameterUnit() {
		// If the animation speed changes or the time format has been changed then
		// this parameter unit will send a signal to the UI controls.
		connect(&ANIM_MANAGER, SIGNAL(speedChanged(int)), this, SIGNAL(formatChanged()));
		connect(&ANIM_MANAGER, SIGNAL(timeFormatChanged()), this, SIGNAL(formatChanged()));
	}

	/// \brief Converts the given string to a time value.
	/// \param valueString This is a string representation of a value as it might have
	///                    been produced by formatValue() or entered by the user.
	/// \return The parsed value in TimeTicks.
	/// \throw Exception when the value could not be parsed.
	/// \sa formatValue()
	virtual FloatType parseString(const QString& valueString) {		
		return (FloatType)ANIM_MANAGER.stringToTime(valueString);
	}
		
	/// \brief Converts a time value to a string.
	/// \param value The time value to be converted. This is in TimeTicks units.
	/// \return The string representation of the value. This can be converted back using parseString().
	/// \sa parseString()
	virtual QString formatValue(FloatType value) {
		return ANIM_MANAGER.timeToString((TimeTicks)value);
	}

	/// \brief Returns the step size used by spinner widgets for this paremeter unit type.
	/// \param currentValue The current value of the spinner. This can be used to make the step size value dependent.
	/// \param upDirection Specifies whether the spinner is dragged in the positive or the negative direction.
	/// \return The numeric step size used by SpinnerWidget for this parameter type. This is in TimeTicks units.
	virtual FloatType stepSize(FloatType currentValue, bool upDirection) { 
		return ANIM_MANAGER.ticksPerFrame();
	}
	
private:
	
	Q_OBJECT
	DECLARE_PLUGIN_CLASS(TimeParameterUnit)
};

///////////////////////////////////////////////////////////////////////////////

/// \def UNITS_MANAGER
/// \brief The predefined instance of the Core::UnitsManager class.
/// 
/// Always use this macro to access the Core::UnitsManager class instance.
#define UNITS_MANAGER		(*UnitsManager::getSingletonInstance())

/**
 * \brief Manages the parameter units.
 * 
 * This is a singleton class with only one predefined instance of this class. 
 * You can access the instance of this class using the UNITS_MANAGER macro.
 * 
 * \author Alexander Stukowski
 * \sa ParameterUnit
 */
class CORE_DLLEXPORT UnitsManager : public QObject
{
	Q_OBJECT
public:

	/// \brief Returns the one and only instance of this class.
	/// \return The predefined instance of the UnitsManager singleton class.
	/// \note You should use the UNITS_MANAGER macro to access the UnitsManager instance instead
	///       of this method.
	inline static UnitsManager* getSingletonInstance() {
		OVITO_ASSERT_MSG(_singletonInstance != NULL, "UnitsManager::getSingletonInstance", "UnitsManager class is not initialized yet.");
		return _singletonInstance;
	}

	/// \brief Returns the special identity parameter unit that does no unit conversion at all and that
	///        formats values as floating-point.
	/// \return Global instance of the default floating-point unit object.
	FloatParameterUnit* floatIdentity() const { 
		CHECK_OBJECT_POINTER(floatIdentityUnit); 
		return floatIdentityUnit; 
	}

	/// \brief Returns the special identity parameter unit that does no unit conversion at all and that
	///        formats values as integer.
	/// \return Global instance of the default integer unit object.
	IntegerParameterUnit* integerIdentity() const { 
		CHECK_OBJECT_POINTER(integerIdentityUnit); 
		return integerIdentityUnit; 
	}

	/// \brief Returns the instance of a parameter unit service.
	/// \param parameterUnitClass Specifies the unit type. This must a ParameterUnit derived class.
	/// \return The global instance of the requested class. The UnitsManager will always return
	///         the same instance of a ParameterUnit class.
	ParameterUnit* getUnit(PluginClassDescriptor* parameterUnitClass);

	/// \brief Returns the instance of a parameter unit service for time values.
	/// \return The global instance of the requested class.
	TimeParameterUnit* getTimeParameterUnit() { return static_object_cast<TimeParameterUnit>(getUnit(PLUGINCLASSINFO(TimeParameterUnit))); }

	/// \brief Returns the instance of a parameter unit service for percentage values.
	/// \return The global instance of the requested class.
	PercentParameterUnit* getPercentParameterUnit() { return static_object_cast<PercentParameterUnit>(getUnit(PLUGINCLASSINFO(PercentParameterUnit))); }

	/// \brief Returns the instance of a parameter unit service for angles.
	/// \return The global instance of the requested class.
	AngleParameterUnit* getAngleParameterUnit() { return static_object_cast<AngleParameterUnit>(getUnit(PLUGINCLASSINFO(AngleParameterUnit))); }

	/// \brief Returns the instance of a parameter unit service for world space distances.
	/// \return The global instance of the requested class.
	WorldParameterUnit* getWorldParameterUnit() { return static_object_cast<WorldParameterUnit>(getUnit(PLUGINCLASSINFO(WorldParameterUnit))); }

private:

	/// The cached unit converters.
	map<PluginClassDescriptor*, ParameterUnit::SmartPtr> units;

	/// The special float identity unit. This field is only used for performance reasons.
	FloatParameterUnit* floatIdentityUnit;

	/// The special integer identity unit. This field is only used for performance reasons.
	IntegerParameterUnit* integerIdentityUnit;

private:
    
	/// This is a singleton class. No public instances allowed.
	UnitsManager();

	/// Initializes the UnitsManager.
	static void initialize() { 
		OVITO_ASSERT(_singletonInstance == NULL);
		_singletonInstance = new UnitsManager();
	}
	
	/// UnitsManager shutdown.
	static void shutdown() {
		delete _singletonInstance;
		_singletonInstance = NULL;
	}
	
	/// The singleton instance of this class.
	static UnitsManager* _singletonInstance;

	friend class ApplicationManager;	
};


};

#endif // __OVITO_PARAMETER_UNIT_H
