/*
 * Copyright 2013 Canonical Ltd.
 * Copyright 2022 Guido Berhoerster <guido+qtorganizer5-eds@berhoerster.name>
 *
 * This file is part of qtorganizer5-eds.
 *
 * contact-service-app is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; version 3.
 *
 * contact-service-app is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 */

#include "qorganizer-eds-fetchrequestdata.h"

#include <QtCore/QDebug>

#include <QtOrganizer/QOrganizerItemFetchRequest>
#include <QtOrganizer/QOrganizerItemCollectionFilter>
#include <QtOrganizer/QOrganizerItemUnionFilter>
#include <QtOrganizer/QOrganizerItemIntersectionFilter>

using namespace QtOrganizer;

FetchRequestData::FetchRequestData(QOrganizerEDSEngine *engine,
                                   const QByteArrayList &sourceIds,
                                   QOrganizerAbstractRequest *req)
    : RequestData(engine, req),
      m_parseListener(0),
      m_currentComponents(0)
{
    // filter collections related with the query
    m_sourceIds = filterSourceIds(sourceIds);
}

FetchRequestData::~FetchRequestData()
{
    delete m_parseListener;

    Q_FOREACH(GSList *components, m_components.values()) {
        g_slist_free_full(components, (GDestroyNotify)g_object_unref);
    }
    m_components.clear();
}

QByteArray FetchRequestData::nextSourceId()
{
    if (m_currentComponents) {
        m_components.insert(m_current, m_currentComponents);
        m_currentComponents = 0;
    }
    m_current = "";
    setClient(0);
    if (m_sourceIds.size()) {
        m_current = m_sourceIds.takeFirst();
        return m_current;
    } else {
        return QByteArray();
    }
}

QByteArray FetchRequestData::nextParentId()
{
    QByteArray nextId;
    if (!m_currentParentIds.isEmpty()) {
        nextId = m_currentParentIds.values().first();
        m_currentParentIds.remove(nextId);
    }
    return nextId;
}

QByteArray FetchRequestData::sourceId() const
{
    return m_current;
}

time_t FetchRequestData::startDate() const
{
    QDateTime startDate = request<QOrganizerItemFetchRequest>()->startDate();
    if (!startDate.isValid()) {
        QDate currentDate = QDate::currentDate();
        startDate.setTime(QTime(0, 0, 0));
        startDate.setDate(QDate(currentDate.year(), 1, 1));
        qWarning() << "Start date is invalid using " << startDate;
    }
    return startDate.toTime_t();
}

time_t FetchRequestData::endDate() const
{
    QDateTime endDate = request<QOrganizerItemFetchRequest>()->endDate();
    if (!endDate.isValid()) {
        QDate currentDate = QDate::currentDate();
        endDate.setTime(QTime(0, 0, 0));
        endDate.setDate(QDate(currentDate.year()+1, 1, 1));
        qWarning() << "End date is invalid using " << endDate;
    }
    return endDate.toTime_t();
}

bool FetchRequestData::hasDateInterval() const
{
    if (!filterIsValid()) {
        return false;
    }

    QDateTime endDate = request<QOrganizerItemFetchRequest>()->endDate();
    QDateTime startDate = request<QOrganizerItemFetchRequest>()->startDate();

    return (endDate.isValid() && startDate.isValid());
}

bool FetchRequestData::filterIsValid() const
{
    return (request<QOrganizerItemFetchRequest>()->filter().type() != QOrganizerItemFilter::InvalidFilter);
}

void FetchRequestData::cancel()
{
    if (m_parseListener) {
        delete m_parseListener;
        m_parseListener = 0;
    }
    RequestData::cancel();
}

void FetchRequestData::compileCurrentIds()
{
    for(GSList *e = m_currentComponents; e != NULL; e = e->next) {
        ICalComponent *icalComp = static_cast<ICalComponent *>(e->data);
        if (e_cal_util_component_has_recurrences (icalComp)) {
            m_currentParentIds.insert(QByteArray(i_cal_component_get_uid(icalComp)));
        }
    }
}

void FetchRequestData::finish(QOrganizerManager::Error error,
                              QOrganizerAbstractRequest::State state)
{
    if (!m_components.isEmpty()) {
        m_parseListener = new FetchRequestDataParseListener(this,
                                                            error,
                                                            state);
        QOrganizerItemFetchRequest *req =  request<QOrganizerItemFetchRequest>();
        if (req) {
            parent()->parseEventsAsync(m_components,
                                       true,
                                       req->fetchHint().detailTypesHint(),
                                       m_parseListener,
                                       SLOT(onParseDone(QList<QtOrganizer::QOrganizerItem>)));

            return;
        }
    }
    finishContinue(error, state);
}

void FetchRequestData::finishContinue(QOrganizerManager::Error error,
                                      QOrganizerAbstractRequest::State state)
{
    if (m_parseListener) {
        m_parseListener->deleteLater();
        m_parseListener = 0;
    }

    Q_FOREACH(GSList *components, m_components.values()) {
        g_slist_free_full(components, (GDestroyNotify)g_object_unref);
    }
    m_components.clear();

    QOrganizerItemFetchRequest *req =  request<QOrganizerItemFetchRequest>();
    if (req) {
        QOrganizerManagerEngine::updateItemFetchRequest(req,
                                                        m_results,
                                                        error,
                                                        state);
    }

    // TODO: emit changeset???
    RequestData::finish(error, state);
}

void FetchRequestData::appendResult(ICalComponent *comp)
{
    m_currentComponents = g_slist_append(m_currentComponents, comp);
}

void FetchRequestData::appendDeatachedResult(ICalComponent *comp)
{
    const gchar *uid;
    ICalTime *rid;

    uid = i_cal_component_get_uid(comp);
    rid = i_cal_component_get_recurrenceid(comp);

    for(GSList *e=m_currentComponents; e != NULL; e = e->next) {
        ICalComponent *ical = static_cast<ICalComponent *>(e->data);
        ICalTime *curRid = i_cal_component_get_recurrenceid(ical);
        bool replace = (g_strcmp0(uid, i_cal_component_get_uid(ical)) == 0) &&
            (i_cal_time_compare(rid, curRid) == 0);
        g_object_unref(curRid);
        if (replace) {

            // replace instance event
            g_object_unref (ical);
            e->data = i_cal_component_clone(comp);
            gchar *time_str = i_cal_time_as_ical_string(rid);
            QByteArray itemId = m_current + '/' +
                QByteArray(uid) + '#' +
                QByteArray(time_str);
            g_free(time_str);
            m_deatachedIds.append(itemId);
            break;
        }
    }

    g_object_unref(rid);
}

int FetchRequestData::appendResults(QList<QOrganizerItem> results)
{
    int count = 0;
    QOrganizerItemFetchRequest *req = request<QOrganizerItemFetchRequest>();
    if (!req) {
        return 0;
    }
    QOrganizerItemFilter filter = req->filter();
    QList<QOrganizerItemSortOrder> sorting = req->sorting();

    Q_FOREACH(QOrganizerItem item, results) {
        if (QOrganizerManagerEngine::testFilter(filter, item)) {
            QOrganizerManagerEngine::addSorted(&m_results, item, sorting);
            count++;
        }
    }
    return count;
}

QString FetchRequestData::dateFilter()
{
    QOrganizerItemFetchRequest *r = request<QOrganizerItemFetchRequest>();
    if (r->filter().type() == QOrganizerItemFilter::InvalidFilter) {
        qWarning("Query for events with invalid filter type");
        return QStringLiteral("");
    }

    QDateTime startDate = r->startDate();
    QDateTime endDate = r->endDate();

    if (!startDate.isValid() ||
        !endDate.isValid()) {
        return QStringLiteral("#t"); // match all
    }

    gchar *startDateStr = isodate_from_time_t(startDate.toTime_t());
    gchar *endDateStr = isodate_from_time_t(endDate.toTime_t());

    QString query = QString("(occur-in-time-range? "
                            "(make-time \"%1\") (make-time \"%2\"))")
            .arg(startDateStr)
            .arg(endDateStr);

    g_free(startDateStr);
    g_free(endDateStr);

    return query;
}

QByteArrayList FetchRequestData::filterSourceIds(const QByteArrayList &sourceIds) const
{
    QByteArrayList result;
    if (filterIsValid()) {
        QOrganizerItemFilter f = request<QOrganizerItemFetchRequest>()->filter();
        QByteArrayList cFilters = sourceIdsFromFilter(f);
        if (cFilters.contains("*") || cFilters.isEmpty()) {
            result = sourceIds;
        } else {
            Q_FOREACH(const QByteArray &f, sourceIds) {
                if (cFilters.contains(f)) {
                    result << f;
                }
            }
        }
    }
    return result;
}

QByteArrayList FetchRequestData::sourceIdsFromFilter(const QOrganizerItemFilter &f) const
{
    QByteArrayList result;

    switch(f.type()) {
    case QOrganizerItemFilter::CollectionFilter:
    {
        QOrganizerItemCollectionFilter cf = static_cast<QOrganizerItemCollectionFilter>(f);
        Q_FOREACH(const QOrganizerCollectionId &id, cf.collectionIds()) {
            result << id.localId();
        }
        break;
    }
    case QOrganizerItemFilter::IntersectionFilter:
    {
        QOrganizerItemIntersectionFilter intersec = static_cast<QOrganizerItemIntersectionFilter>(f);
        Q_FOREACH(const QOrganizerItemFilter &f, intersec.filters()) {
            result << sourceIdsFromFilter(f);
        }
        break;
    }
    case QOrganizerItemFilter::UnionFilter:
        // TODO: better handle union filters, for now they will consider all collections
        result << "*";
        break;
    default:
        break;
    }

    return result;
}

FetchRequestDataParseListener::FetchRequestDataParseListener(FetchRequestData *data,
                                                             QOrganizerManager::Error error,
                                                             QOrganizerAbstractRequest::State state)
    : QObject(0),
      m_data(data),
      m_error(error),
      m_state(state)
{
}

void FetchRequestDataParseListener::onParseDone(QList<QOrganizerItem> results)
{
    m_data->appendResults(results);
    m_data->finishContinue(m_error, m_state);
}
