#   IGraph R package
#   Copyright (C) 2003-2012  Gabor Csardi <csardi.gabor@gmail.com>
#   334 Harvard street, Cambridge, MA 02139 USA
#
#   This program is free software; you can redistribute it and/or modify
#   it under the terms of the GNU General Public License as published by
#   the Free Software Foundation; either version 2 of the License, or
#   (at your option) any later version.
#
#   This program is distributed in the hope that it will be useful,
#   but WITHOUT ANY WARRANTY; without even the implied warranty of
#   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
#   GNU General Public License for more details.
#
#   You should have received a copy of the GNU General Public License
#   along with this program; if not, write to the Free Software
#   Foundation, Inc.,  51 Franklin Street, Fifth Floor, Boston, MA
#   02110-1301 USA
#
###################################################################

###################################################################
# Common functions for plot and tkplot
###################################################################

i.parse.plot.params <- function(graph, params) {
  ## store the arguments
  p <- list(vertex = list(), edge = list(), plot = list())
  for (n in names(params)) {
    if (substr(n, 1, 7) == "vertex.") {
      nn <- substring(n, 8)
      p[["vertex"]][[nn]] <- params[[n]]
    } else if (substr(n, 1, 5) == "edge.") {
      nn <- substring(n, 6)
      p[["edge"]][[nn]] <- params[[n]]
    } else {
      p[["plot"]][[n]] <- params[[n]]
    }
  }

  func <- function(type, name, range = NULL, dontcall = FALSE) {
    if (!type %in% names(p)) {
      stop("Invalid plot option type")
    }
    ret <- function() {
      v <- p[[type]][[name]]
      if (is.function(v) && !dontcall) {
        v <- v(graph)
      }
      if (is.null(range)) {
        return(v)
      } else {
        if (length(v) == 1) {
          return(rep(v, length(range)))
        } else {
          return(rep(v, length.out = max(range) + 1)[[range + 1]])
        }
      }
    }
    if (name %in% names(p[[type]])) {
      ## we already have the parameter
      return(ret())
    } else {
      ## we don't have the parameter, check attributes first
      if (type == "vertex" && name %in% vertex_attr_names(graph)) {
        p[[type]][[name]] <- vertex_attr(graph, name)
        return(ret())
      } else if (type == "edge" && name %in% edge_attr_names(graph)) {
        p[[type]][[name]] <- edge_attr(graph, name)
        return(ret())
      } else if (type == "plot" && name %in% graph_attr_names(graph)) {
        p[[type]][[name]] <- graph_attr(graph, name)
        return(ret())
      } else {
        ## no attributes either, check igraph parameters
        n <- paste(sep = "", type, ".", name)
        v <- igraph_opt(n)
        if (!is.null(v)) {
          p[[type]][[name]] <- v
          return(ret())
        }
        ## no igraph parameter either, use default value
        p[[type]][[name]] <- i.default.values[[type]][[name]]
        return(ret())
      }
    }
  }

  return(func)
}

i.get.edge.labels <- function(graph, edge.labels = NULL) {
  if (is.null(edge.labels)) {
    edge.labels <- rep(NA, ecount(graph))
  }

  edge.labels
}

i.get.labels <- function(graph, labels = NULL) {
  if (is.null(labels)) {
    if ("name" %in% vertex_attr_names(graph)) {
      labels <- vertex_attr(graph, "name")
    } else {
      labels <- seq_len(vcount(graph))
    }
  }
  labels
}

i.get.arrow.mode <- function(graph, arrow.mode = NULL) {
  if (is.character(arrow.mode) &&
    length(arrow.mode) == 1 && substr(arrow.mode, 1, 2) == "a:") {
    arrow.mode <- vertex_attr(graph, substring(arrow.mode, 3))
  }

  if (is.character(arrow.mode)) {
    tmp <- numeric(length(arrow.mode))
    tmp[arrow.mode %in% c("<", "<-")] <- 1
    tmp[arrow.mode %in% c(">", "->")] <- 2
    tmp[arrow.mode %in% c("<>", "<->")] <- 3
    arrow.mode <- tmp
  }

  if (is.null(arrow.mode)) {
    if (is_directed(graph)) {
      arrow.mode <- 2
    } else {
      arrow.mode <- 0
    }
  }

  arrow.mode
}

i.get.main <- function(graph) {
  if (igraph_opt("annotate.plot")) {
    n <- graph$name[1]
    n
  } else {
    ""
  }
}

i.get.xlab <- function(graph) {
  if (igraph_opt("annotate.plot")) {
    paste(vcount(graph), "vertices,", ecount(graph), "edges")
  } else {
    ""
  }
}

igraph.check.shapes <- function(x) {
  xx <- unique(x)
  bad.shapes <- !xx %in% ls(.igraph.shapes)
  if (any(bad.shapes)) {
    bs <- paste(xx[bad.shapes], collapse = ", ")
    stop("Bad vertex shape(s): ", bs, ".")
  }
  x
}

i.postprocess.layout <- function(maybe_layout) {
  if ("layout" %in% names(maybe_layout)) {
    # This branch caters for layout_with_sugiyama, which returns multiple
    # things
    layout <- maybe_layout$layout
  } else {
    # This is the normal path for layout functions that return matrices
    layout <- maybe_layout
  }
  layout
}

#' Optimal edge curvature when plotting graphs
#'
#' If graphs have multiple edges, then drawing them as straight lines does not
#' show them when plotting the graphs; they will be on top of each other. One
#' solution is to bend the edges, with diffenent curvature, so that all of them
#' are visible.
#'
#' `curve_multiple()` calculates the optimal `edge.curved` vector for
#' plotting a graph with multiple edges, so that all edges are visible.
#'
#' @aliases autocurve.edges
#' @param graph The input graph.
#' @param start The curvature at the two extreme edges. All edges will have a
#'   curvature between `-start` and `start`, spaced equally.
#' @return A numeric vector, its length is the number of edges in the graph.
#' @author Gabor Csardi \email{csardi.gabor@@gmail.com}
#' @seealso [igraph.plotting] for all plotting parameters,
#' [plot.igraph()], [tkplot()] and [rglplot()]
#' for plotting functions.
#' @family plot.common
#' @export
#' @importFrom stats ave
#' @keywords graphs
#' @examples
#'
#' g <- make_graph(c(
#'   0, 1, 1, 0, 1, 2, 1, 3, 1, 3, 1, 3,
#'   2, 3, 2, 3, 2, 3, 2, 3, 0, 1
#' ) + 1)
#'
#' curve_multiple(g)
#'
#' set.seed(42)
#' plot(g)
#'
curve_multiple <- function(graph, start = 0.5) {
  el <- apply(as_edgelist(graph, names = FALSE), 1, paste, collapse = ":")
  ave(rep(NA, length(el)), el, FUN = function(x) {
    if (length(x) == 1) {
      return(0)
    } else {
      return(seq(-start, start, length.out = length(x)))
    }
  })
}

.igraph.logo.raster <-
  structure(c(
    16777215L, 16777215L, 16777215L, 16777215L, 16777215L,
    16777215L, 16777215L, 16777215L, 16777215L, 16777215L, 16777215L,
    16777215L, 16777215L, 16777215L, 16777215L, 16777215L, 16777215L,
    16777215L, 16777215L, 16777215L, 16777215L, 16777215L, 16777215L,
    16777215L, 16777215L, 16777215L, 16777215L, 16777215L, 16777215L,
    16777215L, 16777215L, 16777215L, 16777215L, 16777215L, 16777215L,
    16777215L, 16777215L, 16777215L, 16777215L, 16777215L, 16777215L,
    16777215L, 16777215L, 16777215L, 16777215L, 16777215L, 16777215L,
    16777215L, 16777215L, 16777215L, 16777215L, 16777215L, 16777215L,
    16777215L, 16777215L, 16777215L, 16777215L, 16777215L, 16777215L,
    16777215L, 16777215L, 16777215L, 16777215L, 16777215L, 16777215L,
    16777215L, 16777215L, 16777215L, 16777215L, 16777215L, 16777215L,
    16777215L, 16777215L, 16777215L, 16777215L, 16777215L, 16777215L,
    16777215L, 16777215L, 16777215L, 16777215L, 16777215L, 16777215L,
    16777215L, 16777215L, 16777215L, 16777215L, 16777215L, 16777215L,
    16777215L, 16777215L, 16777215L, 16777215L, 16777215L, 16777215L,
    16777215L, 16777215L, 16777215L, 16777215L, 16777215L, 16777215L,
    16777215L, 16777215L, 16777215L, 16777215L, 16777215L, 16777215L,
    16777215L, 16777215L, 16777215L, 16777215L, 16777215L, 16777215L,
    16777215L, 16777215L, 16777215L, 16777215L, 16777215L, 16777215L,
    16777215L, 16777215L, 16777215L, 16777215L, 16777215L, 16777215L,
    16777215L, 16777215L, 16777215L, 16777215L, 16777215L, 16777215L,
    16777215L, 16777215L, 16777215L, 16777215L, 16777215L, 16777215L,
    16777215L, 16777215L, 16777215L, 16777215L, 16777215L, 16777215L,
    16777215L, 16777215L, 16777215L, 16777215L, 16777215L, 16777215L,
    16777215L, 16777215L, 16777215L, 16777215L, 16777215L, 16777215L,
    16777215L, 16777215L, 16777215L, 16777215L, 16777215L, 16777215L,
    16777215L, 16777215L, 16777215L, 16777215L, 16777215L, 16777215L,
    16777215L, 16777215L, 16777215L, 16777215L, 16777215L, 16777215L,
    16777215L, 16777215L, 16777215L, 16777215L, 16777215L, 16777215L,
    16777215L, 16777215L, 16777215L, 16777215L, 16777215L, 16777215L,
    16777215L, 16777215L, 16777215L, 16777215L, 16777215L, 16777215L,
    16777215L, 16777215L, 16777215L, 16777215L, 16777215L, 16777215L,
    16777215L, 16777215L, 16777215L, 16777215L, 16777215L, 16777215L,
    16777215L, 16777215L, 16777215L, 16777215L, 16777215L, 16777215L,
    16777215L, 16777215L, 16777215L, 16777215L, 16777215L, 16777215L,
    16777215L, 16777215L, 16777215L, 16777215L, 16777215L, 16777215L,
    16777215L, 16777215L, 16777215L, 16777215L, 16777215L, 16777215L,
    16777215L, 16777215L, 16777215L, 16777215L, 16777215L, 16777215L,
    16777215L, 16777215L, 16777215L, 16777215L, 16777215L, 16777215L,
    16777215L, 16777215L, 16777215L, 16777215L, 16777215L, 16777215L,
    16777215L, 16777215L, 16777215L, 16777215L, 16777215L, 16777215L,
    16777215L, 16777215L, 16777215L, 16777215L, 16777215L, 16777215L,
    16777215L, 16777215L, 16777215L, 16777215L, 16777215L, 16777215L,
    16777215L, 16777215L, 16777215L, 16777215L, 16777215L, 16777215L,
    16777215L, 16777215L, 16777215L, 16777215L, 16777215L, 16777215L,
    16777215L, 16777215L, 16777215L, 16777215L, 16777215L, 16777215L,
    16777215L, 16777215L, 16777215L, 16777215L, 16777215L, 16777215L,
    16777215L, 16777215L, 16777215L, 16777215L, 16777215L, 16777215L,
    16777215L, 16777215L, 16777215L, 16777215L, 16777215L, 16777215L,
    16777215L, 16777215L, 16777215L, 16777215L, 16777215L, 16777215L,
    16777215L, 16777215L, 16777215L, 16777215L, 16777215L, 16777215L,
    16777215L, 16777215L, 16777215L, 16777215L, 16777215L, 16777215L,
    16777215L, 16777215L, 16777215L, 16777215L, 16777215L, 16777215L,
    16777215L, 16777215L, 16777215L, 16777215L, 16777215L, 16777215L,
    16777215L, 16777215L, 16777215L, 16777215L, 16777215L, 16777215L,
    16777215L, 16777215L, 16777215L, 16777215L, 16777215L, 16777215L,
    16777215L, 16777215L, 16777215L, 16777215L, 16777215L, 16777215L,
    16777215L, 16777215L, 16777215L, 16777215L, 16777215L, 16777215L,
    16777215L, 16777215L, 16777215L, 16777215L, 16777215L, 311332508L,
    1217499541L, 1804702102L, 1066570390L, 211129749L, 16777215L,
    16777215L, 16777215L, 16777215L, 16777215L, 16777215L, 16777215L,
    16777215L, 16777215L, 16777215L, 16777215L, 16777215L, 16777215L,
    16777215L, 16777215L, 16777215L, 16777215L, 16777215L, 16777215L,
    16777215L, 16777215L, 16777215L, 16777215L, 16777215L, 16777215L,
    16777215L, 16777215L, 16777215L, 16777215L, 16777215L, 16777215L,
    16777215L, 16777215L, 16777215L, 16777215L, 16777215L, 16777215L,
    16777215L, 16777215L, 379033495L, 1334940052L, -2104389227L,
    -1450012011L, -2087546218L, 1368494484L, 412456341L, 16777215L,
    16777215L, 16777215L, 16777215L, 16777215L, 16777215L, 16777215L,
    16777215L, 16777215L, 16777215L, 16777215L, 41975936L, 1905496981L,
    -141388906L, -7171435L, -7171435L, -7171435L, -325938283L, 1452380564L,
    16777215L, 16777215L, 16777215L, 16777215L, 16777215L, 16777215L,
    16777215L, 16777215L, 16777215L, 16777215L, 16777215L, 16777215L,
    16777215L, 16777215L, 16777215L, 16777215L, 16777215L, 16777215L,
    16777215L, 16777215L, 16777215L, 16777215L, 16777215L, 16777215L,
    16777215L, 16777215L, 16777215L, 16777215L, 16777215L, 16777215L,
    16777215L, 16777215L, 16777215L, 16777215L, 16777215L, 16777215L,
    41975936L, 1905496981L, -158166379L, -7171435L, -7171435L, -7171435L,
    -7171435L, -7171435L, -141389163L, 1972540052L, 41975936L, 16777215L,
    16777215L, 16777215L, 16777215L, 16777215L, 16777215L, 16777215L,
    16777215L, 16777215L, -2037148780L, -7171435L, -24798561L, -12009013L,
    -13250855L, -11616826L, -24340838L, -7171435L, 1586664085L, 16777215L,
    16777215L, 16777215L, 16777215L, 16777215L, 16777215L, 16777215L,
    16777215L, 16777215L, 16777215L, 16777215L, 16777215L, 16777215L,
    16777215L, 16777215L, 16777215L, 16777215L, 16777215L, 16777215L,
    16777215L, 16777215L, 16777215L, 16777215L, 16777215L, 16777215L,
    16777215L, 16777215L, 16777215L, 16777215L, 16777215L, 16777215L,
    16777215L, 16777215L, 16777215L, 311332508L, -963472747L, -7171435L,
    -7171435L, -7171435L, -7171435L, -7236971L, -7171435L, -7171435L,
    -7171435L, -7171435L, -946695531L, 361927314L, 16777215L, 16777215L,
    16777215L, 16777215L, 16777215L, 16777215L, 16777215L, 798134930L,
    -40791403L, -25321308L, -16061704L, -16715521L, -16715521L, -16715521L,
    -15408144L, -24471653L, -258829418L, 344755353L, 16777215L, 16777215L,
    16777215L, 16777215L, 16777215L, 16777215L, 16777215L, 16777215L,
    16777215L, 16777215L, 16777215L, 16777215L, 16777215L, 16777215L,
    16777215L, 16777215L, 16777215L, 16777215L, 16777215L, 16777215L,
    16777215L, 16777215L, 16777215L, 16777215L, 16777215L, 16777215L,
    16777215L, 16777215L, 16777215L, 16777215L, 16777215L, 16777215L,
    16777215L, -1483500650L, -7171435L, -7171435L, -7824996L, -12858668L,
    -15212050L, -16519427L, -15212050L, -12858668L, -7890531L, -7171435L,
    -7171435L, -1382903147L, 16777215L, 16777215L, 16777215L, 16777215L,
    16777215L, 16777215L, 16777215L, 2056426132L, -7171435L, -13643043L,
    -16715521L, -16715521L, -16715521L, -16715521L, -16715521L, -12139572L,
    -7171435L, 1385337493L, 16777215L, 16777215L, 16777215L, 16777215L,
    16777215L, 16777215L, 16777215L, 16777215L, 16777215L, 16777215L,
    16777215L, 16777215L, 16777215L, 16777215L, 16777215L, 16777215L,
    16777215L, 16777215L, 16777215L, 16777215L, 16777215L, 16777215L,
    16777215L, 16777215L, 16777215L, 16777215L, 16777215L, 16777215L,
    16777215L, 16777215L, 16777215L, 16777215L, 1452380564L, -7171435L,
    -7171435L, -8936279L, -15800587L, -16715521L, -16715521L, -16715521L,
    -16715521L, -16715521L, -15865867L, -9132373L, -7171435L, -7171435L,
    1485934996L, 16777215L, 16777215L, 16777215L, 16777215L, 16777215L,
    16777215L, -1433234795L, -7171435L, -15603981L, -16715521L, -16715521L,
    -16715521L, -16715521L, -16715521L, -14100510L, -7171435L, -2104389227L,
    16777215L, 16777215L, 16777215L, 16777215L, 16777215L, 16777215L,
    16777215L, 16777215L, 16777215L, 16777215L, 16777215L, 16777215L,
    16777215L, 16777215L, 16777215L, 16777215L, 16777215L, 16777215L,
    16777215L, 16777215L, 16777215L, 16777215L, 16777215L, 16777215L,
    16777215L, 16777215L, 16777215L, 16777215L, 16777215L, 16777215L,
    16777215L, 16777215L, -812412011L, -7171435L, -7432808L, -15080979L,
    -16715521L, -16715521L, -16715521L, -16715521L, -16715521L, -16715521L,
    -16715521L, -15277585L, -7498344L, -7171435L, -694971499L, 16777215L,
    16777215L, 16777215L, 16777215L, 16777215L, 16777215L, -1919774060L,
    -7171435L, -14623768L, -16715521L, -16715521L, -16715521L, -16715521L,
    -16715521L, -13120041L, -7171435L, 1704104597L, 16777215L, 16777215L,
    16777215L, 16777215L, 16777215L, 16777215L, 16777215L, 16777215L,
    16777215L, 16777215L, 16777215L, 16777215L, 16777215L, 16777215L,
    16777215L, 16777215L, 16777215L, 16777215L, 16777215L, 16777215L,
    16777215L, 16777215L, 16777215L, 16777215L, 16777215L, 16777215L,
    16777215L, 16777215L, 16777215L, 16777215L, 16777215L, 177838489L,
    -74280299L, -7171435L, -10439750L, -16715521L, -16715521L, -16715521L,
    -16715521L, -16715521L, -16715521L, -16715521L, -16715521L, -16715521L,
    -10701380L, -7171435L, -40725867L, 211129749L, 16777215L, 16777215L,
    16777215L, 16777215L, 16777215L, 1368494484L, -7171435L, -10374471L,
    -16715521L, -16715521L, -16715521L, -16715521L, -16584963L, -9067350L,
    -7171435L, 714248856L, 16777215L, 16777215L, 16777215L, 16777215L,
    16777215L, 16777215L, 16777215L, 16777215L, 16777215L, 16777215L,
    16777215L, 16777215L, 16777215L, 16777215L, 16777215L, 16777215L,
    16777215L, 16777215L, 16777215L, 16777215L, 16777215L, 16777215L,
    16777215L, 16777215L, 16777215L, 16777215L, 16777215L, 16777215L,
    16777215L, 16777215L, 16777215L, 999527315L, -7171435L, -7171435L,
    -12270386L, -16715521L, -16715521L, -16715521L, -16715521L, -16715521L,
    -16715521L, -16715521L, -16715521L, -16715521L, -12531503L, -7171435L,
    -7171435L, 1033015958L, 16777215L, 16777215L, 16777215L, 16777215L,
    16777215L, 33554431L, -1080913258L, -7171435L, -10701636L, -15277329L,
    -16519427L, -14885141L, -9720911L, -7171435L, -1718381676L, 16777215L,
    16777215L, 16777215L, 16777215L, 16777215L, 16777215L, 16777215L,
    16777215L, 16777215L, 16777215L, 16777215L, 16777215L, 16777215L,
    16777215L, 16777215L, 16777215L, 16777215L, 16777215L, 16777215L,
    16777215L, 16777215L, 16777215L, 16777215L, 16777215L, 16777215L,
    16777215L, 16777215L, 16777215L, 16777215L, 16777215L, 16777215L,
    16777215L, 1217499541L, -7171435L, -7171435L, -12793389L, -16715521L,
    -16715521L, -16715521L, -16715521L, -16715521L, -16715521L, -16715521L,
    -16715521L, -16715521L, -13054505L, -7171435L, -7171435L, 1251053972L,
    16777215L, 16777215L, 16777215L, 16777215L, 16777215L, 16777215L,
    479367826L, -929918315L, -7171435L, -7171435L, -7236971L, -7171435L,
    -7171435L, -1366060139L, 227117469L, 16777215L, 16777215L, 16777215L,
    16777215L, 16777215L, 16777215L, 16777215L, 16777215L, 16777215L,
    16777215L, 16777215L, 16777215L, 16777215L, 16777215L, 16777215L,
    16777215L, 16777215L, 16777215L, 16777215L, 16777215L, 16777215L,
    16777215L, 16777215L, 16777215L, 16777215L, 16777215L, 16777215L,
    16777215L, 16777215L, 16777215L, 16777215L, 16777215L, 361927314L,
    -7171435L, -7171435L, -10962753L, -16715521L, -16715521L, -16715521L,
    -16715521L, -16715521L, -16715521L, -16715521L, -16715521L, -16715521L,
    -11289661L, -7171435L, -7171435L, 412456341L, 16777215L, 16777215L,
    16777215L, 16777215L, 16777215L, 16777215L, 16777215L, 1536398230L,
    -7171435L, -778857580L, -1013804395L, -1752067691L, 1334940052L,
    16777215L, 16777215L, 16777215L, 16777215L, 16777215L, 16777215L,
    16777215L, 16777215L, 16777215L, 16777215L, 16777215L, 16777215L,
    16777215L, 16777215L, 16777215L, 16777215L, 16777215L, 16777215L,
    16777215L, 16777215L, 16777215L, 16777215L, 16777215L, 16777215L,
    16777215L, 16777215L, 16777215L, 16777215L, 16777215L, 16777215L,
    16777215L, 16777215L, 16777215L, 16777215L, 16777215L, -544042347L,
    -7171435L, -8086625L, -16061704L, -16715521L, -16715521L, -16715521L,
    -16715521L, -16715521L, -16715521L, -16715521L, -16126983L, -8217439L,
    -7171435L, -426601835L, 16777215L, 16777215L, 16777215L, 16777215L,
    16777215L, 16777215L, 16777215L, 16777215L, -1097690475L, -23948651L,
    579833750L, 16777215L, 16777215L, 16777215L, 16777215L, 16777215L,
    16777215L, 16777215L, 16777215L, 16777215L, 16777215L, 16777215L,
    16777215L, 16777215L, 16777215L, 16777215L, 16777215L, 16777215L,
    16777215L, 16777215L, 16777215L, 16777215L, 16777215L, 16777215L,
    16777215L, 16777215L, 16777215L, 76714645L, 1452446357L, -1986882923L,
    -1785556331L, 1720881813L, 361927317L, 16777215L, 16777215L,
    16777215L, 16777215L, 16777215L, 16777215L, -2070703211L, -7171435L,
    -7171435L, -10570822L, -16649985L, -16715521L, -16715521L, -16715521L,
    -16715521L, -16715521L, -16649985L, -10636101L, -7171435L, -7171435L,
    -2020503147L, 16777215L, 16777215L, 16777215L, 16777215L, 16777215L,
    16777215L, 16777215L, 596808338L, -23948651L, -1114467692L, 16777215L,
    16777215L, 16777215L, 16777215L, 16777215L, 16777215L, 16777215L,
    16777215L, 16777215L, 16777215L, 16777215L, 16777215L, 16777215L,
    16777215L, 16777215L, 16777215L, 16777215L, 16777215L, 16777215L,
    16777215L, 16777215L, 16777215L, 16777215L, 16777215L, 16777215L,
    16777215L, 747803285L, -829255019L, -7171435L, -7171435L, -7171435L,
    -7171435L, -326004074L, 1418891925L, 16777215L, 16777215L, 16777215L,
    16777215L, 16777215L, 127046290L, -728591723L, -7171435L, -7171435L,
    -9786446L, -15603981L, -16715521L, -16715521L, -16715521L, -15538958L,
    -9655375L, -7171435L, -7171435L, -661482859L, 144678815L, 16777215L,
    16777215L, 16777215L, 16777215L, 16777215L, 16777215L, 16777215L,
    -2053991786L, -7171435L, 1502778005L, 16777215L, 16777215L, 16777215L,
    16777215L, 16777215L, 16777215L, 16777215L, 16777215L, 16777215L,
    16777215L, 16777215L, 16777215L, 16777215L, 16777215L, 16777215L,
    16777215L, 16777215L, 16777215L, 16777215L, 16777215L, 16777215L,
    16777215L, 16777215L, 16777215L, 16777215L, 278041237L, -443444587L,
    -7171435L, -10963009L, -14492954L, -15015956L, -12335666L, -24340839L,
    -40725867L, 999461525L, 16777215L, 16777215L, 16777215L, 16777215L,
    16777215L, 848598164L, -225275243L, -7171435L, -7171435L, -7171435L,
    -8347998L, -9720911L, -8348254L, -7171435L, -7171435L, -7171435L,
    -225275243L, 949129878L, 16777215L, 16777215L, 16777215L, 16777215L,
    16777215L, 16777215L, 16777215L, 61516458L, -443379051L, -292384107L,
    127046290L, 16777215L, 16777215L, 16777215L, 16777215L, 16777215L,
    16777215L, 16777215L, 16777215L, 16777215L, 16777215L, 16777215L,
    16777215L, 16777215L, 16777215L, 16777215L, 16777215L, 16777215L,
    16777215L, 16777215L, 16777215L, 16777215L, 16777215L, 16777215L,
    16777215L, 16777215L, -1835887979L, -7171435L, -12008757L, -16715521L,
    -16715521L, -16715521L, -16715521L, -14492954L, -24013930L, -745368939L,
    16777215L, 16777215L, 16777215L, 16777215L, 16777215L, 16777215L,
    546279319L, -1114467692L, -7171435L, -7171435L, -7171435L, -7171435L,
    -7171435L, -7171435L, -7171435L, -1064136043L, 546279319L, 16777215L,
    16777215L, 16777215L, 16777215L, 16777215L, 16777215L, 16777215L,
    16777215L, 1301451413L, -7171435L, -1835822188L, 16777215L, 16777215L,
    16777215L, 16777215L, 16777215L, 16777215L, 16777215L, 16777215L,
    16777215L, 16777215L, 16777215L, 16777215L, 16777215L, 16777215L,
    16777215L, 16777215L, 16777215L, 16777215L, 16777215L, 16777215L,
    16777215L, 16777215L, 16777215L, 16777215L, 16777215L, 16777215L,
    -795700587L, -24340838L, -16519427L, -16715521L, -16715521L,
    -16715521L, -16715521L, -16715521L, -9917004L, -7171435L, 361927317L,
    16777215L, 16777215L, 16777215L, 16777215L, 16777215L, 16777215L,
    33554431L, 1469289365L, -1752067691L, -896363883L, -242052459L,
    -141389163L, -7171435L, -309095531L, 429496729L, 1301451413L,
    -2104389227L, -1215130987L, -879586667L, -1701670251L, 1704104597L,
    798134930L, 75530368L, 16777215L, -1332571499L, -7171435L, 798134930L,
    16777215L, 16777215L, 16777215L, 16777215L, 16777215L, 16777215L,
    16777215L, 16777215L, 16777215L, 16777215L, 16777215L, 16777215L,
    16777215L, 16777215L, 16777215L, 16777215L, 16777215L, 16777215L,
    16777215L, 16777215L, 16777215L, 16777215L, 16777215L, 16777215L,
    16777215L, 16777215L, -174943595L, -9067350L, -16715521L, -16715521L,
    -16715521L, -16715521L, -16715521L, -16715521L, -11420476L, -7171435L,
    999461525L, 16777215L, 16777215L, 16777215L, 16777215L, 16777215L,
    16777215L, 16777215L, 16777215L, 16777215L, 16777215L, 16777215L,
    -1986948715L, -7171435L, -7171435L, -7171435L, -7171435L, -7171435L,
    -7171435L, -7171435L, -7171435L, -7171435L, -7171435L, -158166379L,
    -1517120875L, -74280299L, -879586667L, 16777215L, 16777215L,
    16777215L, 16777215L, 16777215L, 16777215L, 16777215L, 16777215L,
    16777215L, 16777215L, 16777215L, 16777215L, 16777215L, 16777215L,
    16777215L, 16777215L, 16777215L, 16777215L, 16777215L, 16777215L,
    16777215L, 16777215L, 16777215L, 16777215L, 16777215L, 16777215L,
    16777215L, -812477803L, -24340839L, -16519427L, -16715521L, -16715521L,
    -16715521L, -16715521L, -16715521L, -9851469L, -7171435L, 328372885L,
    16777215L, 16777215L, 16777215L, 16777215L, 16777215L, 16777215L,
    16777215L, 16777215L, 16777215L, 16777215L, 261724569L, -1248685419L,
    -7171435L, -7171435L, -7171435L, -7171435L, -7171435L, -7566182L,
    -8355679L, -7171435L, -7171435L, -7171435L, -7171435L, -7171435L,
    -7171435L, -1869376618L, 16777215L, 16777215L, 16777215L, 16777215L,
    16777215L, 16777215L, 16777215L, 16777215L, 16777215L, 16777215L,
    16777215L, 16777215L, 16777215L, 16777215L, 16777215L, 16777215L,
    16777215L, 16777215L, 16777215L, 16777215L, 16777215L, 16777215L,
    16777215L, 16777215L, 16777215L, 16777215L, 16777215L, -1902996843L,
    -7171435L, -11681849L, -16715521L, -16715521L, -16715521L, -16715521L,
    -14166045L, -7236714L, -208498027L, 882086803L, 16777215L, 16777215L,
    16777215L, 16777215L, 16777215L, 16777215L, 16777215L, 16777215L,
    16777215L, 1150456470L, -493710699L, -7171435L, -7171435L, -7303018L,
    -10789959L, -13026608L, -14934812L, -16513548L, -16645131L, -15921426L,
    -14013478L, -11973946L, -8618845L, -7171435L, -7171435L, -23948651L,
    -1768779114L, 144678815L, 16777215L, 16777215L, 16777215L, 16777215L,
    16777215L, 16777215L, 16777215L, 16777215L, 16777215L, 16777215L,
    16777215L, 16777215L, 16777215L, 16777215L, 16777215L, 16777215L,
    16777215L, 16777215L, 16777215L, 16777215L, 16777215L, 16777215L,
    16777215L, 16777215L, 16777215L, 227709589L, -544107883L, -7171435L,
    -10570822L, -13969951L, -14492954L, -11943478L, -24210280L, -23948651L,
    -7171435L, -23948651L, -1517186668L, 529831060L, 16777215L, 16777215L,
    16777215L, 16777215L, 16777215L, 16777215L, 596808338L, -174943595L,
    -7171435L, -7171435L, -8684636L, -14605855L, -16645131L, -16645131L,
    -16645131L, -16645131L, -16645131L, -16645131L, -16645131L, -16645131L,
    -16316174L, -11382080L, -7237226L, -7171435L, -7171435L, -1852665195L,
    16777215L, 16777215L, 16777215L, 16777215L, 16777215L, 16777215L,
    16777215L, 16777215L, 16777215L, 16777215L, 16777215L, 16777215L,
    16777215L, 16777215L, 16777215L, 16777215L, 16777215L, 16777215L,
    16777215L, 16777215L, 16777215L, 16777215L, 16777215L, 16777215L,
    16777215L, 16777215L, 663917205L, -929918315L, -7171435L, -7171435L,
    -7171435L, -7171435L, -393112938L, 1284674197L, 1049661588L,
    -879586667L, -7171435L, -141389163L, -1986948715L, 261724569L,
    16777215L, 16777215L, 16777215L, 41975936L, -1013804395L, -7171435L,
    -7171435L, -11184706L, -16316174L, -16645131L, -16645131L, -16645131L,
    -16645131L, -16645131L, -16645131L, -16645131L, -16645131L, -16645131L,
    -16645131L, -16645131L, -14342690L, -8158305L, -7171435L, -23948651L,
    1066570390L, 16777215L, 16777215L, 16777215L, 16777215L, 16777215L,
    16777215L, 16777215L, 16777215L, 16777215L, 16777215L, 16777215L,
    16777215L, 16777215L, 16777215L, 16777215L, 16777215L, 16777215L,
    16777215L, 16777215L, 16777215L, 16777215L, 16777215L, 16777215L,
    16777215L, 16777215L, 16777215L, 59937429L, 1234342549L, 2140312213L,
    -1936551275L, 1486000789L, 294818453L, 16777215L, 16777215L,
    33554431L, 1519621014L, -527265131L, -7171435L, -342715755L,
    1821545109L, 93952409L, 16777215L, 1922142614L, -7171435L, -7171435L,
    -9868880L, -16645131L, -16645131L, -16645131L, -16645131L, -16645131L,
    -16645131L, -16645131L, -16645131L, -16645131L, -16645131L, -16645131L,
    -16645131L, -16645131L, -16645131L, -14210851L, -7237227L, -7171435L,
    -560819563L, 211129749L, 16777215L, 16777215L, 16777215L, 16777215L,
    16777215L, 16777215L, 16777215L, 16777215L, 16777215L, 16777215L,
    16777215L, 16777215L, 16777215L, 16777215L, 16777215L, 16777215L,
    16777215L, 16777215L, 16777215L, 16777215L, 16777215L, 16777215L,
    16777215L, 16777215L, 16777215L, 16777215L, 16777215L, 16777215L,
    16777215L, 16777215L, 16777215L, 16777215L, 16777215L, 16777215L,
    16777215L, 144678815L, 1989383061L, -258829675L, -7171435L, -644705643L,
    1804767894L, -141389163L, -7171435L, -7829349L, -15658261L, -16645131L,
    -16645131L, -16645131L, -16645131L, -16645131L, -16645131L, -16645131L,
    -16645131L, -16645131L, -16645131L, -16645131L, -16645131L, -16645131L,
    -16645131L, -16645131L, -11184706L, -7171435L, -7171435L, -1785622123L,
    16777215L, 16777215L, 16777215L, 16777215L, 16777215L, 16777215L,
    16777215L, 16777215L, 16777215L, 16777215L, 16777215L, 16777215L,
    16777215L, 16777215L, 16777215L, 16777215L, 16777215L, 16777215L,
    16777215L, 16777215L, 16777215L, 16777215L, 16777215L, 16777215L,
    16777215L, 16777215L, 16777215L, 16777215L, 16777215L, 16777215L,
    16777215L, 16777215L, 16777215L, 16777215L, 16777215L, 16777215L,
    16777215L, 344755353L, -1835822188L, -91057515L, -7171435L, -7171435L,
    -7171435L, -13289772L, -16645131L, -16645131L, -16645131L, -16645131L,
    -16645131L, -16645131L, -16645131L, -16645131L, -16645131L, -16645131L,
    -16645131L, -16645131L, -16645131L, -16645131L, -16645131L, -16645131L,
    -16250383L, -8421470L, -7171435L, -292384107L, 16777215L, 16777215L,
    16777215L, 16777215L, 16777215L, 16777215L, 16777215L, 16777215L,
    16777215L, 16777215L, 16777215L, 16777215L, 16777215L, 16777215L,
    16777215L, 16777215L, 16777215L, 16777215L, 16777215L, 16777215L,
    16777215L, 16777215L, 16777215L, 16777215L, 16777215L, 16777215L,
    16777215L, 16777215L, 16777215L, 16777215L, 16777215L, 16777215L,
    16777215L, 16777215L, 16777215L, 16777215L, 16777215L, 16777215L,
    16777215L, 647271572L, -409824619L, -7171435L, -7566183L, -16513548L,
    -16645131L, -16645131L, -16645131L, -16645131L, -16645131L, -16645131L,
    -16645131L, -16645131L, -16645131L, -16645131L, -16645131L, -16645131L,
    -16645131L, -16645131L, -16645131L, -16645131L, -16645131L, -11447872L,
    -7171435L, -7171435L, 613782933L, 16777215L, 16777215L, 16777215L,
    16777215L, 16777215L, 16777215L, 16777215L, 16777215L, 16777215L,
    16777215L, 16777215L, 16777215L, 16777215L, 16777215L, 16777215L,
    16777215L, 16777215L, 16777215L, 16777215L, 16777215L, 16777215L,
    16777215L, 16777215L, 16777215L, 16777215L, 16777215L, 16777215L,
    16777215L, 16777215L, 16777215L, 16777215L, 16777215L, 16777215L,
    16777215L, 16777215L, 16777215L, 16777215L, 16777215L, 16777215L,
    -460090475L, -7171435L, -9342293L, -16645131L, -16645131L, -16645131L,
    -16645131L, -16645131L, -16645131L, -16645131L, -16645131L, -16645131L,
    -16645131L, -16645131L, -16645131L, -16645131L, -16645131L, -16645131L,
    -16645131L, -16645131L, -16645131L, -13421357L, -7171435L, -7171435L,
    1502778005L, 16777215L, 16777215L, 16777215L, 16777215L, 16777215L,
    16777215L, 16777215L, 16777215L, 16777215L, 16777215L, 16777215L,
    965907093L, -1785556331L, -879586667L, -158166379L, -695037291L,
    -1584229739L, 1435669141L, 16777215L, 16777215L, 16777215L, 16777215L,
    16777215L, 16777215L, 16777215L, 16777215L, 16777215L, 16777215L,
    16777215L, 16777215L, 16777215L, 16777215L, 16777215L, 16777215L,
    16777215L, 16777215L, 16777215L, 16777215L, 412456341L, -7171435L,
    -7171435L, -11184706L, -16645131L, -16645131L, -16645131L, -16645131L,
    -16645131L, -16645131L, -16645131L, -16645131L, -16645131L, -16645131L,
    -16645131L, -16645131L, -16645131L, -16645131L, -16645131L, -16645131L,
    -16645131L, -16645131L, -15263513L, -7171435L, -7171435L, -1903062635L,
    16777215L, 16777215L, 16777215L, 16777215L, 16777215L, 16777215L,
    16777215L, 16777215L, 16777215L, 143823509L, -1936551275L, -40725867L,
    -7171435L, -7171435L, -7171435L, -7171435L, -7171435L, -7171435L,
    -1299017067L, 412258965L, 16777215L, 16777215L, 16777215L, 16777215L,
    16777215L, 16777215L, 16777215L, 16777215L, 16777215L, 16777215L,
    16777215L, 16777215L, 16777215L, 16777215L, 16777215L, 16777215L,
    16777215L, 16777215L, 1200853907L, -7171435L, -7171435L, -12895025L,
    -16645131L, -16645131L, -16645131L, -16645131L, -16645131L, -16645131L,
    -16645131L, -16645131L, -16645131L, -16645131L, -16645131L, -16645131L,
    -16645131L, -16645131L, -16645131L, -16645131L, -16645131L, -16645131L,
    -16579339L, -7566183L, -7171435L, -1114467692L, 16777215L, 16777215L,
    16777215L, 16777215L, 16777215L, 16777215L, 16777215L, 16777215L,
    16777215L, -1231908203L, -7171435L, -7171435L, -7171435L, -8282719L,
    -9655375L, -8544092L, -7236714L, -7171435L, -7171435L, -577596779L,
    194155157L, 16777215L, 16777215L, 16777215L, 16777215L, 16777215L,
    16777215L, 16777215L, 16777215L, 16777215L, 16777215L, 16777215L,
    16777215L, 16777215L, 16777215L, 16777215L, 16777215L, 16777215L,
    747737495L, -7171435L, -7171435L, -11908411L, -16645131L, -16645131L,
    -16645131L, -16645131L, -16645131L, -16645131L, -16645131L, -16645131L,
    -16645131L, -16645131L, -16645131L, -16645131L, -16645131L, -16645131L,
    -16645131L, -16645131L, -16645131L, -16645131L, -15987217L, -7171435L,
    -7171435L, -1483566443L, 16777215L, 16777215L, 16777215L, 16777215L,
    16777215L, 16777215L, 16777215L, 16777215L, 1720881813L, -7171435L,
    -7171435L, -8348254L, -14231324L, -16715521L, -16715521L, -16715521L,
    -15212050L, -9263188L, -7171435L, -7171435L, -1768779115L, 16777215L,
    16777215L, 16777215L, 16777215L, 16777215L, 16777215L, 16777215L,
    16777215L, 16777215L, 16777215L, 16777215L, 16777215L, 16777215L,
    16777215L, 16777215L, 16777215L, 16777215L, 61516458L, -158166379L,
    -7171435L, -10000462L, -16645131L, -16645131L, -16645131L, -16645131L,
    -16645131L, -16645131L, -16645131L, -16645131L, -16645131L, -16645131L,
    -16645131L, -16645131L, -16645131L, -16645131L, -16645131L, -16645131L,
    -16645131L, -16645131L, -14145060L, -7171435L, -7171435L, -91057515L,
    -1315794284L, 1603375510L, 295081622L, 16777215L, 16777215L,
    16777215L, 16777215L, 127046293L, -242052459L, -7171435L, -7629158L,
    -15538958L, -16715521L, -16715521L, -16715521L, -16715521L, -16715521L,
    -16519427L, -8740442L, -7171435L, -23948651L, 747803285L, 16777215L,
    16777215L, 16777215L, 16777215L, 16777215L, 16777215L, 16777215L,
    16777215L, 16777215L, 16777215L, 16777215L, 16777215L, 16777215L,
    16777215L, 16777215L, 16777215L, 16777215L, -963472747L, -7171435L,
    -8158305L, -16645131L, -16645131L, -16645131L, -16645131L, -16645131L,
    -16645131L, -16645131L, -16645131L, -16645131L, -16645131L, -16645131L,
    -16645131L, -16645131L, -16645131L, -16645131L, -16645131L, -16645131L,
    -16645131L, -12237111L, -7171435L, -7171435L, -7171435L, -7171435L,
    -7171435L, -74280299L, -1164865131L, 1754502038L, 412456341L,
    16777215L, 915575445L, -7171435L, -7171435L, -12008757L, -16715521L,
    -16715521L, -16715521L, -16715521L, -16715521L, -16715521L, -16715521L,
    -13773857L, -7171435L, -7171435L, 1720881813L, 16777215L, 16777215L,
    16777215L, 16777215L, 16777215L, 16777215L, 16777215L, 16777215L,
    16777215L, 16777215L, 16777215L, 16777215L, 16777215L, 16777215L,
    16777215L, 16777215L, 16777215L, -1819110763L, -7171435L, -7171435L,
    -15263513L, -16645131L, -16645131L, -16645131L, -16645131L, -16645131L,
    -16645131L, -16645131L, -16645131L, -16645131L, -16645131L, -16645131L,
    -16645131L, -16645131L, -16645131L, -16645131L, -16645131L, -16645131L,
    -9868879L, -7171435L, -74280299L, 1368560277L, -1651338603L,
    -325938539L, -7171435L, -7171435L, -7171435L, -40725867L, -1013804395L,
    -1382903147L, -7171435L, -7171435L, -14100510L, -16715521L, -16715521L,
    -16715521L, -16715521L, -16715521L, -16715521L, -16715521L, -16061960L,
    -7171435L, -7171435L, -1668115819L, 16777215L, 16777215L, 16777215L,
    16777215L, 16777215L, 16777215L, 16777215L, 16777215L, 16777215L,
    16777215L, 16777215L, 16777215L, 16777215L, 16777215L, 16777215L,
    16777215L, 16777215L, 1402180499L, -7171435L, -7171435L, -9539923L,
    -16579339L, -16645131L, -16645131L, -16645131L, -16645131L, -16645131L,
    -16645131L, -16645131L, -16645131L, -16645131L, -16645131L, -16645131L,
    -16645131L, -16645131L, -16645131L, -16645131L, -13816104L, -7171435L,
    -7171435L, -946695531L, 16777215L, 16777215L, 61516458L, 1116967831L,
    -1802333548L, -460090475L, -7171435L, -7171435L, -7171435L, -7171435L,
    -7171435L, -14558233L, -16715521L, -16715521L, -16715521L, -16715521L,
    -16715521L, -16715521L, -16715521L, -16388613L, -7302250L, -7171435L,
    -1433234795L, 16777215L, 16777215L, 16777215L, 16777215L, 16777215L,
    16777215L, 16777215L, 16777215L, 16777215L, 16777215L, 16777215L,
    16777215L, 16777215L, 16777215L, 16777215L, 16777215L, 16777215L,
    16777215L, -1198353770L, -7171435L, -7171435L, -12500278L, -16645131L,
    -16645131L, -16645131L, -16645131L, -16645131L, -16645131L, -16645131L,
    -16645131L, -16645131L, -16645131L, -16645131L, -16645131L, -16645131L,
    -16645131L, -15987217L, -8092514L, -7171435L, -74280299L, 898666645L,
    16777215L, 16777215L, 16777215L, 16777215L, 16777215L, 33554431L,
    949129878L, -1970105706L, -443379050L, -7171435L, -7171435L,
    -12793389L, -16715521L, -16715521L, -16715521L, -16715521L, -16715521L,
    -16715521L, -16715521L, -14558233L, -7171435L, -7171435L, 1972540053L,
    16777215L, 16777215L, 16777215L, 16777215L, 16777215L, 16777215L,
    16777215L, 16777215L, 16777215L, 16777215L, 16777215L, 16777215L,
    16777215L, 16777215L, 16777215L, 16777215L, 16777215L, 16777215L,
    479367826L, -258829675L, -7171435L, -7500391L, -14737438L, -16645131L,
    -16645131L, -16645131L, -16645131L, -16645131L, -16645131L, -16645131L,
    -16645131L, -16645131L, -16645131L, -16645131L, -16645131L, -16447757L,
    -10263627L, -7171435L, -7171435L, -2070703211L, 16777215L, 16777215L,
    16777215L, 16777215L, 16777215L, 16777215L, 16777215L, 16777215L,
    16777215L, 294818453L, -23948651L, -7171435L, -8478812L, -16323334L,
    -16715521L, -16715521L, -16715521L, -16715521L, -16715521L, -16715521L,
    -9917005L, -7171435L, -7171435L, 1083347605L, 16777215L, 16777215L,
    16777215L, 16777215L, 16777215L, 16777215L, 16777215L, 16777215L,
    16777215L, 16777215L, 16777215L, 16777215L, 16777215L, 16777215L,
    16777215L, 16777215L, 16777215L, 16777215L, 16777215L, 1603375510L,
    -7171435L, -7171435L, -7434600L, -12237111L, -16513548L, -16645131L,
    -16645131L, -16645131L, -16645131L, -16645131L, -16645131L, -16645131L,
    -16645131L, -16645131L, -15000603L, -9013337L, -7171435L, -7171435L,
    -778923371L, 109084842L, 16777215L, 16777215L, 16777215L, 16777215L,
    16777215L, 16777215L, 16777215L, 16777215L, 16777215L, 16777215L,
    -1768779115L, -7171435L, -7171435L, -10178634L, -16061960L, -16715521L,
    -16715521L, -16715521L, -16388612L, -11224382L, -7171435L, -7171435L,
    -997027179L, 43160213L, 16777215L, 16777215L, 16777215L, 16777215L,
    16777215L, 16777215L, 16777215L, 16777215L, 16777215L, 16777215L,
    16777215L, 16777215L, 16777215L, 16777215L, 16777215L, 16777215L,
    16777215L, 16777215L, 16777215L, 33554431L, -728591723L, -7171435L,
    -7171435L, -7171435L, -9276502L, -14605855L, -16513549L, -16645131L,
    -16645131L, -16645131L, -16645131L, -16645131L, -15789843L, -12171320L,
    -7368809L, -7171435L, -7171435L, -376270187L, 781226134L, 16777215L,
    16777215L, 16777215L, 16777215L, 16777215L, 16777215L, 16777215L,
    16777215L, 16777215L, 16777215L, 16777215L, 194155157L, -577596779L,
    -7171435L, -7171435L, -7890531L, -10636100L, -12335666L, -11028288L,
    -8413533L, -7171435L, -7171435L, -174943595L, 613585557L, 16777215L,
    16777215L, 16777215L, 16777215L, 16777215L, 16777215L, 16777215L,
    16777215L, 33554431L, 579833750L, 261724569L, 16777215L, 16777215L,
    16777215L, 16777215L, 16777215L, 16777215L, 16777215L, 16777215L,
    160337550L, -1416457579L, -7171435L, -7171435L, -124611948L,
    -7171435L, -7171435L, -7171435L, -7500391L, -9342293L, -11316288L,
    -12171320L, -10263627L, -8355679L, -7171435L, -7171435L, -7171435L,
    -7171435L, -1416457579L, 344755353L, 16777215L, 16777215L, 16777215L,
    16777215L, 16777215L, 16777215L, 16777215L, 16777215L, 16777215L,
    16777215L, 16777215L, 16777215L, 16777215L, 647139989L, -913141099L,
    -7171435L, -7171435L, -7171435L, -7171435L, -7171435L, -7171435L,
    -7171435L, -476933483L, 1150456469L, 16777215L, 16777215L, 16777215L,
    16777215L, 16777215L, 16777215L, 16777215L, 1184010901L, -1131244907L,
    -275606891L, -7171435L, -23948651L, -644705643L, -1768779114L,
    311332508L, 16777215L, 16777215L, 16777215L, 16777215L, 379033495L,
    -929852523L, -7171435L, -23948651L, 2056426132L, 428838809L,
    -1282305642L, -7171435L, -7171435L, -7171435L, -7171435L, -7171435L,
    -7171435L, -7171435L, -7171435L, -7171435L, -7171435L, -325938539L,
    1485934996L, 16777215L, 16777215L, 16777215L, 16777215L, 16777215L,
    16777215L, 16777215L, 16777215L, 16777215L, 16777215L, 16777215L,
    16777215L, 16777215L, 16777215L, 16777215L, 16777215L, 59937429L,
    2039648917L, -711814507L, -40725867L, -7171435L, -7171435L, -510487915L,
    -1752001899L, 261264021L, 16777215L, 16777215L, 16777215L, 16777215L,
    16777215L, 16777215L, 211129749L, -1701670251L, -7171435L, -7171435L,
    -7171435L, -7171435L, -7171435L, -7171435L, -7171435L, -426601835L,
    1234408342L, 16777215L, 16777215L, 697274261L, -544042347L, -7171435L,
    -124611947L, 1485934996L, 16777215L, 16777215L, 16777215L, 1167365268L,
    -2137943659L, -1248619627L, -376270187L, -7171435L, -7171435L,
    -91057515L, -846032235L, -1752067691L, 1653772948L, 395350160L,
    16777215L, 16777215L, 16777215L, 16777215L, 16777215L, 16777215L,
    16777215L, 16777215L, 16777215L, 16777215L, 16777215L, 16777215L,
    16777215L, 16777215L, 16777215L, 16777215L, 16777215L, 16777215L,
    16777215L, 16777215L, 227709589L, 949129877L, 378704533L, 16777215L,
    16777215L, 16777215L, 16777215L, 16777215L, 16777215L, 16777215L,
    16777215L, 16777215L, -1550741099L, -7171435L, -7171435L, -8021089L,
    -11616570L, -13446949L, -12662830L, -10178634L, -7171435L, -7171435L,
    -91057515L, 831689367L, 1133613460L, -275606891L, -7171435L,
    -342715755L, 999527315L, 16777215L, 16777215L, 16777215L, 16777215L,
    16777215L, 16777215L, 16777215L, 16777215L, 529831060L, 865178006L,
    144678815L, 16777215L, 16777215L, 16777215L, 16777215L, 16777215L,
    16777215L, 16777215L, 16777215L, 16777215L, 16777215L, 16777215L,
    16777215L, 16777215L, 16777215L, 16777215L, 16777215L, 16777215L,
    16777215L, 16777215L, 16777215L, 16777215L, 16777215L, 16777215L,
    16777215L, 16777215L, 16777215L, 16777215L, 16777215L, 16777215L,
    16777215L, 16777215L, 16777215L, 16777215L, 16777215L, 16777215L,
    1368494484L, -7171435L, -7171435L, -9851725L, -15996425L, -16715521L,
    -16715521L, -16715521L, -16715521L, -13904672L, -7563622L, -7171435L,
    -476933483L, -91057514L, -7171435L, -644705643L, 613782933L,
    16777215L, 16777215L, 16777215L, 16777215L, 16777215L, 16777215L,
    16777215L, 16777215L, 16777215L, 16777215L, 16777215L, 16777215L,
    16777215L, 16777215L, 16777215L, 16777215L, 16777215L, 16777215L,
    16777215L, 16777215L, 16777215L, 16777215L, 16777215L, 16777215L,
    16777215L, 16777215L, 16777215L, 16777215L, 16777215L, 16777215L,
    16777215L, 16777215L, 16777215L, 16777215L, 16777215L, 16777215L,
    16777215L, 16777215L, 16777215L, 16777215L, 16777215L, 16777215L,
    16777215L, 16777215L, 16777215L, 16777215L, 16777215L, -846032235L,
    -7171435L, -8217439L, -16061704L, -16715521L, -16715521L, -16715521L,
    -16715521L, -16715521L, -16715521L, -12727853L, -7171435L, -7171435L,
    -7171435L, -1030581611L, 311332508L, 16777215L, 16777215L, 16777215L,
    16777215L, 16777215L, 16777215L, 16777215L, 16777215L, 16777215L,
    16777215L, 16777215L, 16777215L, 16777215L, 16777215L, 16777215L,
    16777215L, 16777215L, 16777215L, 16777215L, 16777215L, 16777215L,
    16777215L, 16777215L, 16777215L, 16777215L, 16777215L, 16777215L,
    16777215L, 16777215L, 16777215L, 16777215L, 16777215L, 16777215L,
    16777215L, 16777215L, 16777215L, 16777215L, 16777215L, 16777215L,
    16777215L, 16777215L, 16777215L, 16777215L, 16777215L, 16777215L,
    16777215L, 16777215L, 109084842L, -91057515L, -7171435L, -12139828L,
    -16715521L, -16715521L, -16715521L, -16715521L, -16715521L, -16715521L,
    -16715521L, -16649985L, -7890531L, -7171435L, -695037291L, 109084842L,
    16777215L, 16777215L, 16777215L, 16777215L, 16777215L, 16777215L,
    16777215L, 16777215L, 16777215L, 16777215L, 16777215L, 16777215L,
    16777215L, 16777215L, 16777215L, 16777215L, 16777215L, 16777215L,
    16777215L, 16777215L, 16777215L, 16777215L, 16777215L, 16777215L,
    16777215L, 16777215L, 16777215L, 16777215L, 16777215L, 16777215L,
    16777215L, 16777215L, 16777215L, 16777215L, 16777215L, 16777215L,
    16777215L, 16777215L, 16777215L, 16777215L, 16777215L, 16777215L,
    16777215L, 16777215L, 16777215L, 16777215L, 16777215L, 16777215L,
    831689367L, -7171435L, -7171435L, -13970208L, -16715521L, -16715521L,
    -16715521L, -16715521L, -16715521L, -16715521L, -16715521L, -16715521L,
    -9720911L, -7171435L, -1080913258L, 16777215L, 16777215L, 16777215L,
    16777215L, 16777215L, 16777215L, 16777215L, 16777215L, 16777215L,
    16777215L, 16777215L, 16777215L, 16777215L, 16777215L, 16777215L,
    16777215L, 16777215L, 16777215L, 16777215L, 16777215L, 16777215L,
    16777215L, 16777215L, 16777215L, 16777215L, 16777215L, 16777215L,
    16777215L, 16777215L, 16777215L, 16777215L, 16777215L, 16777215L,
    16777215L, 16777215L, 16777215L, 16777215L, 16777215L, 16777215L,
    16777215L, 16777215L, 16777215L, 16777215L, 16777215L, 16777215L,
    16777215L, 16777215L, 16777215L, 16777215L, 596808338L, -7171435L,
    -7171435L, -13512485L, -16715521L, -16715521L, -16715521L, -16715521L,
    -16715521L, -16715521L, -16715521L, -16715521L, -9197652L, -7171435L,
    -1299017067L, 16777215L, 16777215L, 16777215L, 16777215L, 16777215L,
    16777215L, 16777215L, 16777215L, 16777215L, 16777215L, 16777215L,
    16777215L, 16777215L, 16777215L, 16777215L, 16777215L, 16777215L,
    16777215L, 16777215L, 16777215L, 16777215L, 16777215L, 16777215L,
    16777215L, 16777215L, 16777215L, 16777215L, 16777215L, 16777215L,
    16777215L, 16777215L, 16777215L, 16777215L, 16777215L, 16777215L,
    16777215L, 16777215L, 16777215L, 16777215L, 16777215L, 16777215L,
    16777215L, 16777215L, 16777215L, 16777215L, 16777215L, 16777215L,
    16777215L, 16777215L, 33554431L, -258829675L, -7171435L, -11355453L,
    -16715521L, -16715521L, -16715521L, -16715521L, -16715521L, -16715521L,
    -16715521L, -16192519L, -7498343L, -7171435L, 2089980564L, 16777215L,
    16777215L, 16777215L, 16777215L, 16777215L, 16777215L, 16777215L,
    16777215L, 16777215L, 16777215L, 16777215L, 16777215L, 16777215L,
    16777215L, 16777215L, 16777215L, 16777215L, 16777215L, 16777215L,
    16777215L, 16777215L, 16777215L, 16777215L, 16777215L, 16777215L,
    16777215L, 16777215L, 16777215L, 16777215L, 16777215L, 16777215L,
    16777215L, 16777215L, 16777215L, 16777215L, 16777215L, 16777215L,
    16777215L, 16777215L, 16777215L, 16777215L, 16777215L, 16777215L,
    16777215L, 16777215L, 16777215L, 16777215L, 16777215L, 16777215L,
    16777215L, -1265462635L, -7171435L, -7367273L, -14950677L, -16715521L,
    -16715521L, -16715521L, -16715521L, -16715521L, -16715521L, -10897730L,
    -7171435L, -7171435L, 1049661588L, 16777215L, 16777215L, 16777215L,
    16777215L, 16777215L, 16777215L, 16777215L, 16777215L, 16777215L,
    16777215L, 16777215L, 16777215L, 16777215L, 16777215L, 16777215L,
    16777215L, 16777215L, 16777215L, 16777215L, 16777215L, 16777215L,
    16777215L, 16777215L, 16777215L, 16777215L, 16777215L, 16777215L,
    16777215L, 16777215L, 16777215L, 16777215L, 16777215L, 16777215L,
    16777215L, 16777215L, 16777215L, 16777215L, 16777215L, 16777215L,
    16777215L, 16777215L, 16777215L, 16777215L, 16777215L, 16777215L,
    16777215L, 16777215L, 16777215L, 16777215L, 16777215L, 630296984L,
    -174943595L, -7171435L, -8086625L, -14100766L, -16715521L, -16715521L,
    -16715521L, -16323077L, -11028288L, -7171435L, -7171435L, -1550741099L,
    16777215L, 16777215L, 16777215L, 16777215L, 16777215L, 16777215L,
    16777215L, 16777215L, 16777215L, 16777215L, 16777215L, 16777215L,
    16777215L, 16777215L, 16777215L, 16777215L, 16777215L, 16777215L,
    16777215L, 16777215L, 16777215L, 16777215L, 16777215L, 16777215L,
    16777215L, 16777215L, 16777215L, 16777215L, 16777215L, 16777215L,
    16777215L, 16777215L, 16777215L, 16777215L, 16777215L, 16777215L,
    16777215L, 16777215L, 16777215L, 16777215L, 16777215L, 16777215L,
    16777215L, 16777215L, 16777215L, 16777215L, 16777215L, 16777215L,
    16777215L, 16777215L, 16777215L, 16777215L, 1821545109L, -7171435L,
    -7171435L, -7236971L, -8740186L, -10439750L, -9655375L, -7825252L,
    -7171435L, -7171435L, -476933483L, 277843855L, 16777215L, 16777215L,
    16777215L, 16777215L, 16777215L, 16777215L, 16777215L, 16777215L,
    16777215L, 16777215L, 16777215L, 16777215L, 16777215L, 16777215L,
    16777215L, 16777215L, 16777215L, 16777215L, 16777215L, 16777215L,
    16777215L, 16777215L, 16777215L, 16777215L, 16777215L, 16777215L,
    16777215L, 16777215L, 16777215L, 16777215L, 16777215L, 16777215L,
    16777215L, 16777215L, 16777215L, 16777215L, 16777215L, 16777215L,
    16777215L, 16777215L, 16777215L, 16777215L, 16777215L, 16777215L,
    16777215L, 16777215L, 16777215L, 16777215L, 16777215L, 16777215L,
    16777215L, 16777215L, 16777215L, 1385337493L, -376270187L, -7171435L,
    -7171435L, -7171435L, -7171435L, -7171435L, -7171435L, -1332571499L,
    395350160L, 16777215L, 16777215L, 16777215L, 16777215L, 16777215L,
    16777215L, 16777215L, 16777215L, 16777215L, 16777215L, 16777215L,
    16777215L, 16777215L, 16777215L, 16777215L, 16777215L, 16777215L,
    16777215L, 16777215L, 16777215L, 16777215L, 16777215L, 16777215L,
    16777215L, 16777215L, 16777215L, 16777215L, 16777215L, 16777215L,
    16777215L, 16777215L, 16777215L, 16777215L, 16777215L, 16777215L,
    16777215L, 16777215L, 16777215L, 16777215L, 16777215L, 16777215L,
    16777215L, 16777215L, 16777215L, 16777215L, 16777215L, 16777215L,
    16777215L, 16777215L, 16777215L, 16777215L, 16777215L, 16777215L,
    16777215L, 16777215L, 344755353L, 1922142614L, -1533898091L,
    -728591723L, -1080913258L, -1903062635L, 1284805780L, 16777215L,
    16777215L, 16777215L, 16777215L, 16777215L, 16777215L, 16777215L,
    16777215L, 16777215L, 16777215L, 16777215L, 16777215L, 16777215L,
    16777215L, 16777215L, 16777215L, 16777215L, 16777215L, 16777215L,
    16777215L, 16777215L, 16777215L, 16777215L, 16777215L, 16777215L,
    16777215L, 16777215L, 16777215L, 16777215L, 16777215L, 16777215L,
    16777215L, 16777215L, 16777215L, 16777215L, 16777215L, 16777215L,
    16777215L, 16777215L, 16777215L, 16777215L, 16777215L, 16777215L,
    16777215L, 16777215L, 16777215L, 16777215L, 16777215L, 16777215L,
    16777215L, 16777215L, 16777215L, 16777215L, 16777215L, 16777215L,
    16777215L, 16777215L, 16777215L, 16777215L, 16777215L, 16777215L,
    16777215L, 16777215L, 16777215L, 16777215L, 16777215L, 16777215L,
    16777215L, 16777215L, 16777215L, 16777215L, 16777215L, 16777215L,
    16777215L, 16777215L, 16777215L, 16777215L, 16777215L, 16777215L,
    16777215L, 16777215L, 16777215L, 16777215L, 16777215L, 16777215L,
    16777215L, 16777215L, 16777215L, 16777215L, 16777215L, 16777215L,
    16777215L, 16777215L, 16777215L, 16777215L, 16777215L, 16777215L,
    16777215L, 16777215L, 16777215L, 16777215L, 16777215L, 16777215L,
    16777215L, 16777215L, 16777215L, 16777215L, 16777215L, 16777215L,
    16777215L, 16777215L, 16777215L, 16777215L, 16777215L, 16777215L,
    16777215L, 16777215L, 16777215L, 16777215L, 16777215L, 16777215L,
    16777215L, 16777215L, 16777215L, 16777215L, 16777215L, 16777215L,
    16777215L, 16777215L, 16777215L, 16777215L, 16777215L, 16777215L,
    16777215L, 16777215L, 16777215L, 16777215L, 16777215L, 16777215L,
    16777215L, 16777215L, 16777215L, 16777215L, 16777215L, 16777215L,
    16777215L, 16777215L, 16777215L, 16777215L, 16777215L, 16777215L,
    16777215L, 16777215L, 16777215L, 16777215L, 16777215L, 16777215L,
    16777215L, 16777215L, 16777215L, 16777215L, 16777215L, 16777215L,
    16777215L, 16777215L, 16777215L, 16777215L, 16777215L, 16777215L,
    16777215L, 16777215L, 16777215L, 16777215L, 16777215L, 16777215L,
    16777215L, 16777215L, 16777215L, 16777215L, 16777215L, 16777215L,
    16777215L, 16777215L, 16777215L, 16777215L, 16777215L, 16777215L,
    16777215L, 16777215L, 16777215L, 16777215L, 16777215L, 16777215L,
    16777215L, 16777215L, 16777215L, 16777215L, 16777215L, 16777215L,
    16777215L, 16777215L, 16777215L, 16777215L, 16777215L, 16777215L,
    16777215L, 16777215L, 16777215L, 16777215L, 16777215L, 16777215L,
    16777215L, 16777215L, 16777215L, 16777215L, 16777215L, 16777215L,
    16777215L, 16777215L, 16777215L, 16777215L, 16777215L, 16777215L,
    16777215L, 16777215L, 16777215L, 16777215L, 16777215L, 16777215L,
    16777215L, 16777215L, 16777215L, 16777215L, 16777215L, 16777215L,
    16777215L, 16777215L, 16777215L, 16777215L, 16777215L, 16777215L,
    16777215L, 16777215L, 16777215L, 16777215L, 16777215L, 16777215L,
    16777215L, 16777215L, 16777215L, 16777215L, 16777215L, 16777215L,
    16777215L, 16777215L, 16777215L, 16777215L, 16777215L, 16777215L,
    16777215L, 16777215L, 16777215L, 16777215L, 16777215L, 16777215L,
    16777215L, 16777215L, 16777215L, 16777215L, 16777215L, 16777215L,
    16777215L, 16777215L, 16777215L, 16777215L, 16777215L, 16777215L,
    16777215L, 16777215L, 16777215L, 16777215L, 16777215L, 16777215L,
    16777215L, 16777215L, 16777215L, 16777215L, 16777215L, 16777215L,
    16777215L, 16777215L, 16777215L, 16777215L, 16777215L, 16777215L,
    16777215L, 16777215L, 16777215L, 16777215L, 16777215L, 16777215L,
    16777215L, 16777215L, 16777215L, 16777215L, 16777215L, 16777215L,
    16777215L, 16777215L, 16777215L, 16777215L, 16777215L, 16777215L,
    16777215L, 16777215L, 16777215L, 16777215L, 16777215L, 16777215L,
    16777215L, 16777215L, 16777215L, 16777215L, 16777215L, 16777215L,
    16777215L, 16777215L, 16777215L, 16777215L, 16777215L, 16777215L,
    16777215L, 16777215L, 16777215L, 16777215L, 16777215L, 16777215L,
    16777215L, 16777215L, 16777215L, 16777215L, 16777215L, 16777215L,
    16777215L, 16777215L, 16777215L, 16777215L, 16777215L, 16777215L,
    16777215L, 16777215L, 16777215L, 16777215L, 16777215L, 16777215L,
    16777215L, 16777215L, 16777215L, 16777215L, 16777215L, 16777215L,
    16777215L, 16777215L, 16777215L, 16777215L, 16777215L, 16777215L,
    16777215L, 16777215L, 16777215L, 16777215L, 16777215L, 16777215L,
    16777215L, 16777215L, 16777215L, 16777215L, 16777215L, 16777215L,
    16777215L, 16777215L, 16777215L, 16777215L, 16777215L, 16777215L,
    16777215L, 16777215L, 16777215L, 16777215L, 16777215L, 16777215L,
    16777215L, 16777215L, 16777215L, 16777215L, 16777215L, 16777215L,
    16777215L, 16777215L, 16777215L, 16777215L, 16777215L, 16777215L,
    16777215L, 16777215L, 16777215L, 16777215L, 16777215L, 16777215L,
    16777215L, 16777215L, 16777215L, 16777215L, 16777215L, 16777215L,
    16777215L, 16777215L, 16777215L, 16777215L, 16777215L, 16777215L,
    16777215L, 16777215L, 16777215L, 16777215L, 16777215L, 16777215L,
    16777215L, 16777215L, 16777215L, 16777215L, 16777215L, 16777215L,
    16777215L, 16777215L, 16777215L
  ), .Dim = c(64L, 64L), class = "nativeRaster", channels = 4L)

i.vertex.default <- list(
  color = 1,
  size = 15,
  size2 = 15,
  label = i.get.labels,
  label.degree = -pi / 4,
  label.color = "darkblue",
  label.dist = 0,
  label.family = "serif",
  label.font = 1,
  label.cex = 1,
  frame.color = "black",
  frame.width = 1,
  shape = "circle",
  pie = 1,
  pie.color = list(c(
    "white", "lightblue", "mistyrose",
    "lightcyan", "lavender", "cornsilk"
  )),
  pie.angle = 45,
  pie.density = -1,
  pie.lty = 1,
  raster = .igraph.logo.raster
)

i.edge.default <- list(
  color = "darkgrey",
  label = i.get.edge.labels,
  lty = 1,
  width = 1,
  loop.angle = 0,
  loop.angle2 = 0,
  label.family = "serif",
  label.font = 1,
  label.cex = 1,
  label.color = "darkblue",
  label.x = NULL,
  label.y = NULL,
  arrow.size = 1,
  arrow.mode = i.get.arrow.mode,
  curved = curve_multiple,
  arrow.width = 1
)

i.plot.default <- list(
  palette = categorical_pal(8),
  layout = layout_nicely,
  margin = c(0, 0, 0, 0),
  rescale = TRUE,
  asp = 1,
  frame = FALSE,
  main = i.get.main,
  sub = "",
  xlab = i.get.xlab,
  ylab = ""
)

i.default.values <- new.env()

i.default.values[["vertex"]] <- i.vertex.default
i.default.values[["edge"]] <- i.edge.default
i.default.values[["plot"]] <- i.plot.default
