/*
 * Mimic the 'mpp' request which is used on Cray PBSPro pbs_mom nodes.
 */
#include "basil_alps.h"

/* Don't bother with realloc, use huge buffer (rosa: min 330kb) */
#define MPP_BUFFER_LENGTH	(512 * 1024)
#define MPP_MIN_ROOM		512	/* must contain full node string */

/*
 * Convert a struct basil_response structure returned from an inventory
 * method call into a form that will be passed to the scheduler.
 *
 * Format = Version^Nodes^Reservations
 * where...
 * Version = 1.0
 * Nodes = Node;Node;...
 * Reservations = Reservation;Reservation;...
 * where...
 * Node = node_id:arch:state:role:name:processors:memory:labels
 * Reservation = rsvn_id:user_name:account_name
 * where...
 * processors = ordinal|arch|clock_mhz|allocations,...
 * memory = type|page_size_kb|page_count|allocations,...
 * labels = name|type|disposition,...
 * where...
 * processor allocations = rsvn_id&rsvn_id&...
 * memory allocations = rsvn_id/page_count&rsvn_id/page_count&...
 *
 * A function in the scheduler takes this format and assigns the
 * values to a structure that is used to evaluate jobs requesting
 * mpp resources. We could pass the XML directly to the scheduler,
 * but the payload can get a bit heavy when systems contain tens
 * of thousands of compute nodes. We could compress the data on
 * the fly to reduce much of the payload, which would probably be
 * more elegant. This works for now.
 */

/*
 * Node list
 */
static size_t print_proc(char *p, const struct basil_node_processor *proc)
{
	ssize_t len = 0;

	if (proc == NULL)
		return 0;

	if (proc->next) {
		len += print_proc(p, proc->next);
		len += sprintf(p + len, ",");
	}
	len += sprintf(p + len, "%d|%u|%d|",	proc->ordinal,
						proc->arch,
						proc->clock_mhz);
	if (proc->allocation)
		len += sprintf(p + len, "%u", proc->allocation->rsvn_id);
	return len;
}

static size_t print_mema(char *p, const struct basil_mem_alloc *mema)
{
	ssize_t len = 0;

	if (mema == NULL)
		return 0;
	if (mema->next) {
		len += print_mema(p, mema->next);
		len += sprintf(p + len, "&");
	}
	len += sprintf(p + len, "%u/%u",	mema->rsvn_id,
						mema->page_count);
	return len;
}

static size_t print_mem(char *p, const struct basil_node_memory *mem)
{
	ssize_t len = 0;

	if (mem == NULL)
		return 0;
	if (mem->next) {
		len += print_mem(p, mem->next);
		len += sprintf(p + len, ",");
	}

	len += sprintf(p + len, "%u|%u|%u|",	mem->type,
			    			mem->page_size_kb,
						mem->page_count);
	len += print_mema(p + len, mem->a_head);
	return len;
}

static size_t print_label(char *p, const struct basil_label *label)
{
	ssize_t len = 0;

	if (label == NULL)
		return 0;
	if (label->next) {
		len += print_label(p, label->next);
		len += sprintf(p + len, ",");
	}

	len += sprintf(p + len, "%s|%u|%u",	label->name,
			    			label->type,
						label->disp);
	return len;
}

static size_t print_node(char *p, const struct basil_node *node)
{
	ssize_t len = 0;

	if (node == NULL)
		return 0;

	if (node->next) {
		len += print_node(p, node->next);
		len += sprintf(p + len, ";");
	}
	if (len + MPP_MIN_ROOM > MPP_BUFFER_LENGTH)
		errx(1, "Memory of size %dkb too small",
		     MPP_BUFFER_LENGTH / 1024);

	len += sprintf(p + len, "%u:%u:%u:%u:%s:",	node->node_id,
							node->arch,
							node->state,
							node->role,
							node->name);
	len += print_proc(p + len, node->seg_head->proc_head);
	len += sprintf(p + len, ":");
	len += print_mem(p + len, node->seg_head->mem_head);
	len += sprintf(p + len, ":");
	len += print_label(p + len, node->seg_head->lbl_head);
	return len;
}

/*
 * Reservation list
 */
static size_t print_rsvn(char *p, const struct basil_rsvn *rsvn)
{
	ssize_t len = 0;

	if (rsvn == NULL)
		return 0;

	if (rsvn->next) {
		len += print_rsvn(p, rsvn->next);
		len += sprintf(p + len, ";");
	}

	if (len + MPP_MIN_ROOM > MPP_BUFFER_LENGTH)
		errx(1, "Memory of size %dkb too small",
		     MPP_BUFFER_LENGTH / 1024);

	len += sprintf(p + len, "%u:%s:%s",	rsvn->rsvn_id,
						rsvn->user_name,
						rsvn->account_name);
	return len;
}

const char *mpp_inventory(void)
{
	static char mpp_buf[MPP_BUFFER_LENGTH];
	/*
	 * Note: run this query only in Basil 1.0 (BV_1_0), since the MPP
	 * inventory was not designed to handle NUMA Node Segments.
	 */
	const enum basil_version version = BV_1_0;
	struct basil_inventory *inv;
	ssize_t len;

	inv = get_full_inventory(version);
	if (inv == NULL)
		return NULL;

	len = sprintf(mpp_buf, "%s^", bv_names[version]);
	len += print_node(mpp_buf + len, inv->f->node_head);
	len += sprintf(mpp_buf + len, "^");
	len += print_rsvn(mpp_buf + len, inv->f->rsvn_head);

	free_inv(inv);

	return mpp_buf;
}
