/*  File: pl-vmi.h

    This file provides the definition of type code, and also includes
    macros to generate other types of output on a per-VMI basis. For
    more examples, see the files pl-wam.c and pl-codetable.c.
*/

#ifndef _PL_VMI_H
#define _PL_VMI_H
#define _IN_PL_VMI_H

/* Including autogenerated vmi-metadata.h. Global defines:
 * VM_SIGNATURE 0x1234abcd
 * FOREACH_VMI_CALL(sep,f,...) f(D_BREAK,...) sep() f(I_NOP,...) sep() f(H_ATOM,...) sep() ...
 * FOREACH_VMH_CALL(sep,f,...) f(wakeup,...) sep() f(retry,...) sep() f(h_const,...) sep() ...
 * 
 * Per-instruction defines, assuming the following in pl-vmi.c:
 *   VMI(INSTR_NAME, I_FLAGS, 2, (CA1_ARGTYPE,CA1_ARGTYPE))
 * 
 * VMLCASE_INSTR_NAME instr_name
 * VMIDECL_INSTR_NAME INSTR_NAME, I_FLAGS, 2, (CA1_ARGTYPE,CA1_ARGTYPE)
 * VMIFLAGS_INSTR_NAME I_FLAGS
 * VMIARGCOUNT_INSTR_NAME 2
 * VMIARGTYPES_INSTR_NAME CA1_ARGTYPE,CA1_ARGTYPE
 * 
 * Per-helper defines, assuming the following:
 *   VMH(helper_name, 2, (int, Code), (n, p))
 * 
 * VMLCASE_helper_name helper_name
 * VMHDECL_helper_name helper_name, 2, (int, Code), (n, p)
 * VMHARGCOUNT_helper_name 2
 * VMHARGTYPES_helper_name int, Code
 * VMHARGNAMES_helper_name n, p
 */
#include <vmi-metadata.h>

#define VMI_DECL(n)	VMIDECL_ ## n
#define VMI_NAME(n)	n
#define VMI_LOWER(n)	VMLCASE_ ## n
#define VMI_STRING(n)	A_STRINGIFY(VMI_LOWER(n))
#define VMI_FLAGS(n)	VMIFLAGS_ ## n
#define VMI_ARGCOUNT(n)	VMIARGCOUNT_ ## n
#define VMI_ARGTYPES(n)	VMIARGTYPES_ ## n

#define VMH_DECL(n)	VMHDECL_ ## n
#define VMH_NAME(n)	n
#define VMH_ARGCOUNT(n)	VMHARGCOUNT_ ## n
#define VMH_ARGTYPES(n)	VMHARGTYPES_ ## n
#define VMH_ARGNAMES(n)	VMHARGNAMES_ ## n

/* Macros to create valid C identifiers given a VMI/VMH name */
#define VMI_IDENT(n)	instr_ ## n
#define VMH_IDENT(n)	helper_ ## n

/* Macros to expand a code template once for each VMI/VMH definition.
 * Commas are effectively used as "escape characters" to introduce one of:
 * (a) a callout - a bare macro name that will be passed the name of the VMI, or
 * (b) a literal, in parentheses, that can contain commas, or
 * (c) an empty parameter, which is replaced by a single comma.
 * 
 * Example usage:
 * FOREACH_VMI(T_EMPTY,
 *   int ,VMI_IDENT,(Code PC,, ,(DECL_LD void *ctx),);
 * )
 * 
 * This would expand to:
 * int instr_D_BREAK(DECL_LD Code PC, void *ctx);
 * ... and so on. Note the double-comma acting as an escape for a single comma, and
 * the ,(...), production to escape an entire sequence with optional commas.
 */
#define FOREACH_VMI(sep,...)		FOREACH_VMI_CALL(sep, M_TEMPLATE, __VA_ARGS__)
#define FOREACH_VMH(sep,...)		FOREACH_VMH_CALL(sep, M_TEMPLATE, __VA_ARGS__)

/* Call macro f, passing it the same parameters as VMI() or VMH(), separating each
 * with sep().
 */
#define FOREACH_VMIDECL_CALL(sep,f)	FOREACH_VMI_CALL(sep, VMIDECL_CALL, f)
#define FOREACH_VMHDECL_CALL(sep,f)	FOREACH_VMH_CALL(sep, VMHDECL_CALL, f)

#define VMIDECL_CALL(n, f)		A_CALL(f, VMI_DECL(n))
#define VMHDECL_CALL(n, f)		A_CALL(f, VMH_DECL(n))

typedef enum
{ FOREACH_VMI(T_COMMA,
    ,VMI_NAME,
  ),
  VMI_END_LIST
} vmi;

#define I_HIGHEST ((int)VMI_END_LIST)
/* VM_SIGNATURE is defined in pl-vmi.h */

#undef _IN_PL_VMI_H
#endif /* _PL_VMI_H */