/*******************************************************************************
* localinput.cpp: Local streams
*-------------------------------------------------------------------------------
* (c)1999-2001 VideoLAN
* $Id: localinput.cpp,v 1.1 2001/11/29 16:11:42 bozo Exp $
*
* Authors: Arnaud de Bossoreille de Ribou <bozo@via.ecp.fr>
*
* This program is free software; you can redistribute it and/or
* modify it under the terms of the GNU General Public License
* as published by the Free Software Foundation; either version 2
* of the License, or (at your option) any later version.
*
* This program is distributed in the hope that it will be useful,
* but WITHOUT ANY WARRANTY; without even the implied warranty of
* MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
* GNU General Public License for more details.
*
* You should have received a copy of the GNU General Public License
* along with this program; if not, write to the Free Software
* Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.
*
*-------------------------------------------------------------------------------
*
*******************************************************************************/


//------------------------------------------------------------------------------
// Preamble
//------------------------------------------------------------------------------
#include "../../core/defs.h"

#include <sys/time.h>
#include <unistd.h>

#include "../../core/core.h"

#include "../../mpeg/mpeg.h"
#include "../../mpeg/ts.h"

#include "../../server/program.h"
#include "../../server/buffer.h"
#include "../../server/output.h"
#include "../../server/channel.h"
#include "../../server/broadcast.h"
#include "../../server/request.h"
#include "../../server/input.h"
#include "../../server/tsstreamer.h"

#include "../../mpeg/reader.h"
#include "../../mpeg/converter.h"

#include "localinput.h"


//------------------------------------------------------------------------------
// Library declaration
//------------------------------------------------------------------------------
#ifdef __PLUGIN__
GENERATE_LIB_ARGS(C_LocalInputModule, handle);
#endif


//------------------------------------------------------------------------------
// Builtin declaration
//------------------------------------------------------------------------------
#ifdef __BUILTIN__
C_Module* NewBuiltin_localinput(handle hLog)
{
  return new C_LocalInputModule(hLog);
}
#endif


/*******************************************************************************
* C_LocalInput class
********************************************************************************
*
*******************************************************************************/

//------------------------------------------------------------------------------
// Constructor
//------------------------------------------------------------------------------
C_LocalInput::C_LocalInput(C_Module* pModule,
                           const C_String& strName) : C_Input(pModule,
                                                              strName)
{
}


//------------------------------------------------------------------------------
// Destructor
//------------------------------------------------------------------------------
C_LocalInput::~C_LocalInput()
{
}


//------------------------------------------------------------------------------
// Initialization
//------------------------------------------------------------------------------
void C_LocalInput::OnInit()
{
  C_Application* pApp = C_Application::GetApp();
  ASSERT(pApp);

  // Get the path to the config file
  C_String strSettingName = GetName() + ".ConfigPath";
  m_strConfigPath = pApp->GetSetting(strSettingName, "");

  // Load settings
  m_cSettings.Load(m_strConfigPath + "/input.cfg", false);

  m_strFilesPath = m_cSettings.GetSetting("Input.FilesPath", "");
  if(    m_strFilesPath.Length()
      && m_strFilesPath[m_strFilesPath.Length() -1] != '/')
    m_strFilesPath += "/";

  // Build the program list
  C_String strPgrmCount = m_cSettings.GetSetting("Input.PgrmCount", "0");
  unsigned int uiPgrmCount = strPgrmCount.ToInt();

  if(uiPgrmCount)
  {
    for(unsigned int ui = 1; ui <= uiPgrmCount; ui++)
    {
      C_String strName = m_cSettings.GetSetting(C_String(ui) + ".Name",
                                                C_String("Pgrm") + ui);
      C_String* pStr = new C_String(strName);
      m_vProgramNames.Add(pStr);
    }
  }
  else
  {
    Log(m_hLog, LOG_WARN,
        "The \"PgrmCount\" variable isn't specified in the file \"" +
        m_strConfigPath + "/input.cfg\"");
  }
}


//------------------------------------------------------------------------------
// Destruction
//------------------------------------------------------------------------------
void C_LocalInput::OnDestroy()
{
}


//------------------------------------------------------------------------------
//
//------------------------------------------------------------------------------
C_List<C_Program> C_LocalInput::OnGetAvailablePgrms()
{
  C_List<C_Program> cPgrmList;

  for(unsigned int ui = 0; ui < m_vProgramNames.Size(); ui++)
  {
    C_Program* pProgram = new C_Program(ui + 1, m_vProgramNames[ui]);
    ASSERT(pProgram);
    cPgrmList.PushEnd(pProgram);
  }

  return cPgrmList;
}


//------------------------------------------------------------------------------
// Start the reception of the given program
//------------------------------------------------------------------------------
void C_LocalInput::OnStartStreaming(C_Broadcast* pBroadcast)
{
  ASSERT(pBroadcast);

  // We choose a TS packet buffer able to store up to 3s of stream at 8 Mbits/s
  C_NetList* pTsProvider = new C_NetList(3*3*797);

  // The netlist must be at least of the size of the Reader buffer +
  // the size of the output buffer + 2 to be sure that there are always free
  // packets in it
  const C_Channel* pChannel = pBroadcast->GetChannel();
  ASSERT(pTsProvider->Capacity() - pChannel->GetBuffCapacity() - 2 > 0);
  unsigned int uiSize = pTsProvider->Capacity() -pChannel->GetBuffCapacity()-2;
  C_SyncFifo* pBuffer;

  // Get the type of the program
  unsigned int uiId =
        m_vProgramNames.Find(pBroadcast->GetProgram()->GetName()) + 1;
  ASSERT(uiId > 0);
  C_String strType = m_cSettings.GetSetting(C_String(uiId) + ".Type",
                                            "Mpeg2-TS");

  C_String strReaderType;
  C_String strConverterType;

  // Specific behaviour depends on the type
  if(strType == "Mpeg1-PS")             // Mpeg 1 - Program Stream
  {
    // Update the size of the buffer and create it
    uiSize -= 2;
    pBuffer = new C_SyncFifo(uiSize);
    // Reader configuration
    strReaderType = "file";
    C_String strFile = m_strFilesPath +
                       m_cSettings.GetSetting(C_String(uiId)+".FileName", "");
    pBroadcast->SetOption("filename", strFile);

    // Converter configuration
    strConverterType = "ps2ts";
    pBroadcast->SetOption("mpegversion", "1");
    pBroadcast->SetOption("preparse", "1");
  }
  else if(strType == "Mpeg2-PS")        // Mpeg 2 - Program Stream
  {
    // Update the size of the buffer and create it
    uiSize -= 2;
    pBuffer = new C_SyncFifo(uiSize);
    // Reader configuration
    strReaderType = "file";
    C_String strFile = m_strFilesPath +
                       m_cSettings.GetSetting(C_String(uiId)+".FileName", "");
    pBroadcast->SetOption("filename", strFile);

    // Converter configuration
    strConverterType = "ps2ts";
    pBroadcast->SetOption("mpegversion", "2");
    pBroadcast->SetOption("preparse", "1");
  }
  else if(strType == "Mpeg2-TS")        // Mpeg 2 - Transport Stream
  {
    // Update the size of the buffer and create it
    pBuffer = new C_SyncFifo(uiSize);
    // Reader configuration
    strReaderType = "file";
    C_String strFile = m_strFilesPath +
                       m_cSettings.GetSetting(C_String(uiId)+".FileName", "");
    pBroadcast->SetOption("filename", strFile);

    // Converter configuration
    strConverterType = "ts2ts";
  }
  else if(strType == "Dvd")             // DVD device (Mpeg2 PS)
  {
    // Update the size of the buffer and create it
    uiSize -= 2;
    pBuffer = new C_SyncFifo(uiSize);
    // Reader configuration
    strReaderType = "dvd";
    C_String strDevice = m_cSettings.GetSetting(C_String(uiId)+".Device", "");
    pBroadcast->SetOption("device", strDevice);

    // Converter configuration
    strConverterType = "ps2ts";
    pBroadcast->SetOption("mpegversion", "2");
    pBroadcast->SetOption("preparse", "0");
  }
  else
  {
    delete pTsProvider;
    throw E_Exception(GEN_ERR, "Unsupported or unknown type : " + strType);
  }

  C_MpegReader* pReader;

  // Create the converter
  C_MpegReaderModule* pReaderModule = (C_MpegReaderModule*)
                                C_Application::GetModuleManager()
                                        ->GetModule("mpegreader",
                                                    strReaderType);

  if(pReaderModule)
  {
    pReader = pReaderModule->NewMpegReader(pBroadcast);
    ASSERT(pReader);
  }
  else
  {
    throw E_Exception(GEN_ERR,
                      "Module mpegreader:" + strConverterType +
                      " not present");
  }

  C_MpegConverter* pConverter;

  // Create the converter
  C_MpegConverterModule* pConverterModule = (C_MpegConverterModule*)
                                C_Application::GetModuleManager()
                                        ->GetModule("mpegconverter",
                                                    strConverterType);

  if(pConverterModule)
  {
    C_MpegConverterConfig cConfig;
    cConfig.m_hLog = m_hLog;
    cConfig.m_pBroadcast = pBroadcast;
    cConfig.m_pReader = pReader;
    cConfig.m_pTsProvider = pTsProvider;
    cConfig.m_pBuffer = pBuffer;
    cConfig.m_pEventHandler = m_pEventHandler;
    pConverter = pConverterModule->NewMpegConverter(cConfig);
    ASSERT(pConverter);
  }
  else
  {
    throw E_Exception(GEN_ERR,
                      "Module mpegconverter:" + strConverterType +
                      " not present");
  }

  pReader->SetConverter(pConverter);

  // Create the streamer
  C_TsStreamer* pStreamer = new C_TsStreamer(m_hLog, pBroadcast,
                                             pTsProvider, pBuffer,
                                             m_pEventHandler);
  ASSERT(pStreamer);

  m_cConverters.Add(pBroadcast, pConverter);
  m_cStreamers.Add(pBroadcast, pStreamer);

  try
  {
    pConverter->Create();
    pStreamer->Create();
  }
  catch(E_Exception e)
  {
    pConverter->Stop();
    pStreamer->Stop();

    // Unregister the 2 thread and delete them
    m_cConverters.Delete(pBroadcast);
    m_cStreamers.Delete(pBroadcast);

    throw E_Exception(GEN_ERR, "unable to start streaming of program "+
                      pBroadcast->GetProgram()->GetName(), e);
  }
}


//------------------------------------------------------------------------------
// Resume the reception of the given program
//------------------------------------------------------------------------------
void C_LocalInput::OnResumeStreaming(C_Broadcast* pBroadcast)
{
  ASSERT(pBroadcast);

  // Find the converter that receive the pgrm
  C_MpegConverter* pConverter = m_cConverters.Get(pBroadcast);
  ASSERT(pConverter);

  // Resume the thread
  try
  {
    pConverter->Resume();
  }
  catch(E_Exception e)
  {
    throw E_Exception(GEN_ERR, "Unable to resume streaming of program "+
                      pBroadcast->GetProgram()->GetName(), e);
  }
}


//------------------------------------------------------------------------------
// Suspend the reception of the given program
//------------------------------------------------------------------------------
void C_LocalInput::OnSuspendStreaming(C_Broadcast* pBroadcast)
{
  ASSERT(pBroadcast);

  // Find the converter that receive the pgrm
  C_MpegConverter* pConverter = m_cConverters.Get(pBroadcast);
  ASSERT(pConverter);

  C_String strPgrmName = pBroadcast->GetProgram()->GetName();

  // Suspend the thread
  try
  {
    pConverter->Suspend();
  }
  catch(E_Exception e)
  {
    throw E_Exception(GEN_ERR, "Unable to resume streaming of program "+
                      pBroadcast->GetProgram()->GetName(), e);
  }
}


//------------------------------------------------------------------------------
// Stop the reception of the given program
//------------------------------------------------------------------------------
void C_LocalInput::OnStopStreaming(C_Broadcast* pBroadcast)
{
  ASSERT(pBroadcast);

  // Find the reader and the streamer that receive the pgrm
  C_MpegConverter* pConverter = m_cConverters.Remove(pBroadcast);
  ASSERT(pConverter);
  C_TsStreamer* pStreamer = m_cStreamers.Remove(pBroadcast);
  ASSERT(pStreamer);

  // Stop the threads
  try
  {
    pConverter->Stop();
    delete pConverter;
    pStreamer->Stop();
    delete pStreamer;
  }
  catch(E_Exception e)
  {
    throw E_Exception(GEN_ERR, "Unable to stop streaming of program "+
                      pBroadcast->GetProgram()->GetName(), e);
  }
}


