/*****************************************************************************
 * config.c: mini-VideoLAN Channel Server, configuration system
 *****************************************************************************
 * Copyright (C) 2001 VideoLAN
 * $Id: config.c,v 1.6 2002/10/12 19:45:31 marcari Exp $
 *
 * Authors: Christophe Massiot <massiot@via.ecp.fr>
 *          Marc Ariberti <marcari@via.ecp.fr>
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111, USA.
 *****************************************************************************/

#include <sys/types.h>
#include <sys/stat.h>
#include <sys/socket.h>
#include <stdio.h>
#include <unistd.h>
#include <sys/time.h>
#include <sys/wait.h>
#include <netinet/in.h>
#include <stdlib.h>
#include <string.h>
#include <stdarg.h>
#include <syslog.h>
#include <errno.h>
#include <arpa/inet.h>
#include <signal.h>
#include <time.h>

#include "vlcs.h"
#include "logger.h"
#include "config.h"

C_Config::C_Config(C_Logger &logger) 
    : logger(logger)
{
    m_pszConfigFile = strdup(CHANNEL_CONF);
}

C_Config::~C_Config()
{
    DestroyList();
    free(m_pszConfigFile);
}

/*
 *  Destroy the linked list
 */
void C_Config::DestroyList( void )
{
    channel_t * p_channel = p_channels;
    channel_t * p_next = NULL;

    while( p_channel != NULL )
    {
        p_next = p_channel->next;
        free(p_channel);
        p_channel = p_next;
    }
    p_channels = NULL;

    logger << C_Logger::INFO << "List of channels cleared" << C_Logger::END;
}

char * C_Config::RemoveLF( char * str )
{
    int i_pos = 0;

    while( str[i_pos] )
    {
        if( str[i_pos] == '\n' )
        {
            str[i_pos] = ' ';
        }
        i_pos++;
    }
    return str;
}

/*
 * Adds the item to the list
 */
void C_Config::AddList( char * psz_item )
{
    channel_t ** pp_cur = &p_channels;
    channel_t * p_new;

    /* finds the end of the linked list */
    while( (*pp_cur) != NULL ) 
    {
        pp_cur = &((*pp_cur)->next);
    }
   
    
    p_new = (channel_t *)malloc( sizeof( channel_t ) );
    *pp_cur = p_new;
    
    strncpy( p_new->psz_vlc_config, psz_item, 255 );
    p_new->psz_vlc_config[255] = 0;
    p_new->next = NULL;

    RemoveLF( p_new->psz_vlc_config);
    
    i_channel_max++;

    logger << C_Logger::INFO << "Adding channel " << i_channel_max << " : " 
            << p_new->psz_vlc_config << C_Logger::END;
}

time_t C_Config::GetModificationTime( void )
{
    struct stat buf;

    stat( m_pszConfigFile, &buf);

    return buf.st_mtime;
}

void C_Config::ReadFile( void )
{
    char psz_vlc_command[2048];
    char psz_buffer[2048];
    FILE * p_config;

    DestroyList();

    st_mtime = GetModificationTime();
    
    p_config = fopen( m_pszConfigFile, "r" );
    if( p_config == NULL )
    {
        logger << C_Logger::ALERT << "Cannot fopen " << m_pszConfigFile 
                << " : "<< strerror(errno) << C_Logger::END;
        exit( -1 );
    }

    i_channel_max = -1;

    while( fgets( psz_buffer, sizeof( psz_buffer ), p_config ) != NULL )
    {
        if( psz_buffer[0] != '#' && psz_buffer[0] != '\0' )
        {
            sscanf(psz_buffer, "%s", psz_vlc_command);
            AddList(psz_vlc_command);
        }
    }

    fclose( p_config );
}

void C_Config::CheckUpdate()
{
    if( st_mtime != GetModificationTime() )
    {
        ReadFile();
    }
}

channel_t * C_Config::GetChannel( int i_channel )
{
    channel_t * p_channel = p_channels;
    while( i_channel > 0 )
    {
        p_channel = p_channel->next;
        if( p_channel == NULL )
        {
            return NULL;
        }
        i_channel--;
    }
    return p_channel;
}

int C_Config::GetChannelNumber( void )
{
    return i_channel_max + 1;
}

void C_Config::SetConfigFile(char * psz_config_file)
{
    free(m_pszConfigFile);
    m_pszConfigFile = strdup(psz_config_file);
}
